package com.uva.rafael.tfg_goniometer.data.entities;

import android.content.ContentValues;

import com.uva.rafael.tfg_goniometer.data.Contract;

/**
 * Esta clase modela a la <tt>Entity</tt> "Medición" de la Base de Datos.
 *
 * <p>Se encarga de proveer un método para insertar una "Medición" en la Base de Datos, pasando cada
 * uno de los atributos que componen esta clase a un formato adecuado, por medio de un objeto de
 * tipo <tt>ContentValues</tt>.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see ArticulacionInvolucrada
 * @see Movimiento
 * @see ContentValues
 * @see com.uva.rafael.tfg_goniometer.data.Contract.MeasurementEntry
 * @version 1.0
 */

public class Medicion {
    // Fecha y hora en la que se realizó la medición
    private final String fecha_hora;
    // Lectura obtenida del goniómetro
    private final double lectura;
    /*
     * Articulación involucrada en la medición.
     *
     * Esta compuesta por la articulación en sí, mas el lado en el que se encuentra la misma.
     */
    private final ArticulacionInvolucrada articulacion;
    /*
     * Movimiento realizado por la articulación en la medición.
     *
     * Esta compuesto por el movimiento en sí, mas el modo en el que se realizó el mismo (si el
     * paciente lo realizó solo, con ayuda...)
     */
    private final Movimiento movimiento;
    // Nombre del paciente al que se ha realizado la medición
    private final String nombrePaciente;
    // ID del paciente al que se ha realizado la medición
    private final int idPaciente;

    /**
     * Constructor principal de la clase
     *
     * @param fecha_hora     Fecha y hora
     * @param lectura        Lectura obtenida del goniómetro
     * @param articulacion   Articulación involucrada
     * @param movimiento     Movimiento realizado por la articulación
     * @param nombrePaciente Nombre del paciente
     */
    public Medicion(String fecha_hora, double lectura, ArticulacionInvolucrada articulacion,
                    Movimiento movimiento, String nombrePaciente, int idPaciente) {
        this.fecha_hora = fecha_hora;
        this.lectura = lectura;
        this.articulacion = articulacion;
        this.movimiento = movimiento;
        this.nombrePaciente = nombrePaciente;
        this.idPaciente = idPaciente;
    }

    /**
     * Método que se encarga de dar un formato adecuado a cada uno de los atributos de la clase,
     * para, posteriormente, ser insertados en la Base de Datos.
     *
     * @return values <tt>ContentValues</tt> con los atributos de la Medición ya formateados
     */
    public ContentValues toContentValues() {
        // ContentValues a devolver con los atributos de la medición ya formateados
        ContentValues values = new ContentValues();

        // Relacionar el campo de la tabla en la BD con el atributo que corresponde de la clase
        values.put(Contract.MeasurementEntry.FECHA_HORA, fecha_hora);
        values.put(Contract.MeasurementEntry.LECTURA, lectura);
        /*
         * Para la obtención del ContentValues de la Articulacion y el Movimiento de la misma, se
         * utilizan los métodos "toContentValues" de las clases correspondientes, ya que, estas
         * clases poseen atributos de tipo "Enum", y su tratamiento es mas complejo desde esta
         * clase
         */
        articulacion.toContentValues(values);
        movimiento.toContentValues(values);
        values.put(Contract.MeasurementEntry.NOMBRE_PACIENTE, nombrePaciente);
        values.put(Contract.MeasurementEntry.ID_PACIENTE, idPaciente);

        // Devolver el ContentValues con los datos en el formato correspondiente
        return values;
    }
}
