package com.uva.rafael.tfg_goniometer.data.enums;

import com.uva.rafael.tfg_goniometer.R;

import xdroid.enumformat.EnumFormat;
import xdroid.enumformat.EnumString;

/**
 * Esta clase es una clase de tipo <tt>Enum</tt> que modela el lado del cuerpo humano (con respecto
 * al plano sagital) en el que se encuentra una articulación, o bien, si se trata de la columna
 * vertebral.
 *
 * <p>Ademas, destacar que esta clase se encarga de proveer estos valores tanto en español como en
 * inglés.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @version 1.0
 */
public enum Lado {
    // Lados del cuerpo humano con respecto al plano sagital
    @EnumString(R.string.lado_izq)
    IZQUIERDO,
    @EnumString(R.string.lado_dcho)
    DERECHO,
    // Columna vertebral
    @EnumString(R.string.columna)
    COLUMNA;

    /**
     * Método que se encarga de devolver el <tt>Enum</tt> asociado con el valor de tipo
     * <tt>String</tt> recibido como parámetro. En caso de que no encuentre ningún <tt>Enum</tt> que
     * se corresponda con este valor, lanza una excepción de tipo <tt>IllegalArgumentException</tt>,
     * emulando el comportamiento del método <tt>valueOf</tt>.
     *
     * @param value Valor a comparar con los valores de los distintos <tt>Enum</tt> de la clase
     * @return <tt>Enum</tt> asociado a ese valor
     */
    public static Lado findByValue(String value) {
        for (Lado lado : values()) {
            // Se compara el valor recibido como parámetro con los valores de cada uno de los Enum
            if (EnumFormat.getInstance().format(lado).equalsIgnoreCase(value))
                /*
                 * Para llevar a cabo esta tarea, se emplea el método "format" de la clase
                 * "EnumFormat", perteneciente a la biblioteca con el mismo nombre
                 */
                return lado;
        }
        /*
         * Si no se ha localizado ningún Enum que coincida con el valor recibido como parámetro,
         * se lanza una excepción
         */
        throw new IllegalArgumentException();
    }

    /**
     * Método <tt>toString()</tt> que se encarga de devolver el String con el valor seleccionado
     * por el usuario, al introducir una nueva medición.
     */
    @Override
    public String toString() {
        /*
         * Para llevar a cabo esta tarea, se emplea el método "format" de la clase "EnumFormat",
         * perteneciente a la biblioteca con el mismo nombre empleada para la i18n de los tipos Enum
         */
        return EnumFormat.getInstance().format(this);
    }
}
