package com.uva.rafael.tfg_goniometer.data.enums;

import android.content.ContentValues;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.data.Contract;

import xdroid.enumformat.EnumFormat;
import xdroid.enumformat.EnumString;

/**
 * Esta clase es una clase de tipo <tt>Enum</tt> que modela los modos de movimiento de una
 * articulación.
 *
 * <p>Se encarga de proveer un método para insertar un "Modo de movimiento" en la Base de Datos,
 * pasando cada uno de los atributos que componen esta clase a un formato adecuado, por medio de un
 * objeto de tipo <tt>ContentValues</tt>.</p>
 *
 * <p>Como dato a señalar, si se ha seleccionado el valor OTRO, lo que se introduzca en el
 * <tt>EditText</tt> se almacenara en <tt>otroModo</tt>.</p>
 *
 * <p>Por último, destacar que esta clase se encarga de proveer estos valores tanto en español como
 * en inglés.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see com.uva.rafael.tfg_goniometer.data.Contract.MeasurementEntry
 * @version 1.0
 */
public enum Modo {
    // Principales movimientos
    @EnumString(R.string.activo)
    ACTIVO, // Movimiento realizado exclusivamente por el usuario
    @EnumString(R.string.activo_asistido)
    ACTIVO_ASISTIDO, // Movimiento realizado por el usuario con cierta ayuda
    @EnumString(R.string.pasivo)
    PASIVO, // Movimiento realizado por el fisioterapeuta o profesional pertinente
    // Este valor se utiliza para introducir un modo de movimiento que no esté en esta lista
    @EnumString(R.string.otro)
    OTRO;

    // Texto introducido por el usuario cuando ha seleccionado el valor OTRO
    private String otroModo = "";

    /**
     * Método que se encarga de devolver el <tt>Enum</tt> asociado con el valor de tipo
     * <tt>String</tt> recibido como parámetro. En caso de que no encuentre ningún <tt>Enum</tt> que
     * se corresponda con este valor, lanza una excepción de tipo <tt>IllegalArgumentException</tt>,
     * emulando el comportamiento del método <tt>valueOf</tt>.
     *
     * @param value Valor a comparar con los valores de los distintos <tt>Enum</tt> de la clase
     * @return <tt>Enum</tt> asociado a ese valor
     */
    public static Modo findByValue(String value) {
        for (Modo modo : values()) {
            // Se compara el valor recibido como parámetro con los valores de cada uno de los Enum
            if (EnumFormat.getInstance().format(modo).equalsIgnoreCase(value))
                /*
                 * Para llevar a cabo esta tarea, se emplea el método "format" de la clase
                 * "EnumFormat", perteneciente a la biblioteca con el mismo nombre
                 */
                return modo;
        }
        /*
         * Si no se ha localizado ningún Enum que coincida con el valor recibido como parámetro,
         * se lanza una excepción
         */
        throw new IllegalArgumentException();
    }

    /**
     * "Constructor" de la clase, que se emplea cuando el usuario ha seleccionado el valor "OTRO" en
     * el Modo de Movimiento.
     *
     * @param otroModo Nombre del modo de movimiento introducido por el usuario en el
     *                 <tt>EditText</tt>
     * @return Instancia de <tt>Modo</tt> con el valor "OTRO" en <tt>modo</tt> y el valor recibido
     * como parámetro en <tt>otroModo</tt>.
     */
    public Modo getOtroModo(String otroModo) {
        this.otroModo = otroModo;

        return this;
    }

    /**
     * Método que se encarga de dar un formato adecuado a cada uno de los atributos de la clase,
     * para, posteriormente, ser insertados en la Base de Datos.
     *
     * @param values <tt>ContentValues</tt>, recibido desde <tt>Medicion</tt>, al que añadir la
     *               información del <tt>Modo</tt> de movimiento
     */
    public void toContentValues(ContentValues values) {
        // Relacionar el campo de la tabla en la BD con el atributo que corresponde de la clase
        values.put(Contract.MeasurementEntry.MODO, this.toString());
        values.put(Contract.MeasurementEntry.OTRO_MODO, otroModo);
    }

    /**
     * Método <tt>toString()</tt> que se encarga de devolver el String con el valor seleccionado
     * por el usuario, al introducir una nueva medición.
     */
    @Override
    public String toString() {
        /*
         * Para llevar a cabo esta tarea, se emplea el método "format" de la clase "EnumFormat",
         * perteneciente a la biblioteca con el mismo nombre empleada para la i18n de los tipos Enum
         */
        return EnumFormat.getInstance().format(this);
    }
}
