package com.uva.rafael.tfg_goniometer.presenter;

import android.app.Fragment;
import android.content.pm.ActivityInfo;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.interfaces.PresenterFunctions;
import com.uva.rafael.tfg_goniometer.model.MainModel;
import com.uva.rafael.tfg_goniometer.view.MainActivity;
import com.uva.rafael.tfg_goniometer.view.fragments.HomeFragment;
import com.uva.rafael.tfg_goniometer.view.fragments.MedicionFragment;
import com.uva.rafael.tfg_goniometer.view.fragments.PacientesFragment;
import com.uva.rafael.tfg_goniometer.view.fragments.UltimasMedicionesFragment;

/**
 * Este es el Presentador asociado al <tt>Fragment</tt> inicial de la aplicación
 * (<tt>HomeFragment</tt>). Se encarga de llevar a cabo toda la lógica asociada a las acciones del
 * usuario realizadas en la IU.
 *
 * <p>En concreto, se encarga de realizar la "configuración inicial" del <tt>Fragment</tt>, que,
 * consiste en actualizar el item del <tt>NavigationView</tt> al primer ítem del mismo, fijar la
 * orientación de la pantalla a "Portrait", para evitar que se pueda girar y fijar el <tt>Toolbar</tt>
 * como <tt>SupportActionBar</tt>.</p>
 *
 * <p>Ademas de eso, se encarga de crear los diferentes <tt>Fragment</tt> en función del botón sobre
 * el que haya pulsado el usuario.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see HomeFragment
 * @see MainModel
 * @see MedicionFragment
 * @see UltimasMedicionesFragment
 * @see PacientesFragment
 * @version 1.0
 */
@SuppressWarnings({"FieldCanBeLocal", "unused"})
public class HomePresenter implements PresenterFunctions, PresenterFunctions.HomeFunctions {

    private final MainModel model;
    // Referencias al fragmento (Vista) con el que esta asociado y al Modelo de la aplicación
    private HomeFragment fragment;

    /**
     * Constructor principal de la clase
     *
     * @param fragment Fragmento (Vista) con la que mantiene una relación 1-a-1.
     * @param model    Modelo (único) de la aplicación.
     */
    public HomePresenter(Fragment fragment, MainModel model) {
        this.fragment = (HomeFragment) fragment;
        this.model = model;
    }

    /**
     * Método que realiza las operaciones iniciales cuando se crea el <tt>Fragment HomeFragment</tt>.
     *
     * Se encarga de marcar el primer ítem del <tt>NavigationView</tt> (correspondiente a
     * <tt>HomeFragment</tt>), fijar la orientación de la pantalla en "Portrait" y de utilizar el
     * <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>.
     *
     * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
     */
    @Override
    public void setUpInitialSettings(Toolbar toolbar) {
        // Marcar el primer ítem del Navigationview
        ((MainActivity) fragment.getActivity()).setNavigationItem(0);

        // Fijar la orientación de la pantalla en "Portrait"
        fragment.getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);

        // Utilizar el toolbar como SupportActionBar
        AppCompatActivity activity = ((AppCompatActivity) fragment.getActivity());

        activity.setSupportActionBar(toolbar);

        if (activity.getSupportActionBar() != null)
            activity.getSupportActionBar().setHomeAsUpIndicator(R.mipmap.ic_menu);

        activity.getSupportActionBar().setDisplayHomeAsUpEnabled(true);
        activity.getSupportActionBar().setTitle("");
    }

    /**
     * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
     * "NUEVA MEDICIÓN" en el <tt>HomeFragment</tt>.
     *
     * Crea un nuevo <tt>Fragment</tt> de tipo <tt>MedicionFragment</tt>.
     */
    @Override
    public void onNuevaMedicionClicked() {
        // "NUEVA MEDICIÓN"
        MedicionFragment medicionFragment = new MedicionFragment();

        fragment.getFragmentManager()
                .beginTransaction()
                .replace(R.id.content_frame, medicionFragment)
                .addToBackStack("MedicionFragment")
                .commit();
    }

    /**
     * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
     * "ULTIMAS 10 MEDICIONES" en el <tt>HomeFragment</tt>.
     *
     * Crea un nuevo <tt>Fragment</tt> de tipo <tt>UltimasMedicionesFragment</tt>.
     */
    @Override
    public void onUltimasMedicionesClicked() {
        // "ULTIMAS 10 MEDICIONES"
        UltimasMedicionesFragment ultimasMedicionesFragment = new UltimasMedicionesFragment();

        fragment.getFragmentManager()
                .beginTransaction()
                .replace(R.id.content_frame, ultimasMedicionesFragment)
                .addToBackStack("UltimasMedicionesFragment")
                .commit();
    }

    /**
     * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
     * "PERFILES DE PACIENTES" en el <tt>HomeFragment</tt>.
     *
     * Crea un nuevo <tt>Fragment</tt> de tipo <tt>PacientesFragment</tt>.
     */
    @Override
    public void onPerfilPacientesClicked() {
        // "PERFILES DE PACIENTES"
        PacientesFragment pacientesFragment = new PacientesFragment();

        fragment.getFragmentManager()
                .beginTransaction()
                .replace(R.id.content_frame, pacientesFragment)
                .addToBackStack("PacientesFragment")
                .commit();
    }

    /**
     * Perform any final cleanup before an activity is destroyed.
     *
     * Se encarga de liberar la referencia al <tt>Fragment</tt> con el que esta asociado.
     */
    @Override
    public void onDestroy() {
        fragment = null;
    }
}
