package com.uva.rafael.tfg_goniometer.presenter;

import android.app.Fragment;
import android.content.pm.ActivityInfo;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.support.v4.graphics.drawable.RoundedBitmapDrawable;
import android.support.v4.graphics.drawable.RoundedBitmapDrawableFactory;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.interfaces.PresenterFunctions;
import com.uva.rafael.tfg_goniometer.model.MainModel;
import com.uva.rafael.tfg_goniometer.view.MainActivity;
import com.uva.rafael.tfg_goniometer.view.fragments.SobreAppFragment;

/**
 * Este es el Presentador asociado al <tt>Fragment</tt> inicial de la aplicación
 * (<tt>SobreAppFragment</tt>). Se encarga de llevar a cabo toda la lógica asociada a las acciones
 * del usuario realizadas en la IU.
 *
 * <p>En concreto, se encarga de realizar la "configuración inicial" del <tt>Fragment</tt>, que,
 * consiste en actualizar el item del <tt>NavigationView</tt> al quinto ítem del mismo, fijar la
 * orientación de la pantalla a "Portrait", para evitar que se pueda girar, fijar el <tt>Toolbar</tt>
 * como <tt>SupportActionBar</tt>, y, a partir del icono principal de la aplicación (imagen con
 * formato rectangular), transformarlo en una imagen con un borde circular y enviárselo a la Vista
 * (<tt>SobreAppFragment</tt>) para que se lo asigne al objeto <tt>ImageView</tt> que contiene el
 * icono de la aplicación.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informática (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see SobreAppFragment
 * @see MainModel
 * @see Bitmap
 * @see RoundedBitmapDrawable
 * @version 1.0
 */
@SuppressWarnings({"FieldCanBeLocal", "unused"})
public class SobreAppPresenter implements PresenterFunctions, PresenterFunctions.SobreAppFunctions {

    private final MainModel model;
    // Referencias al fragmento (Vista) con el que está asociado y al Modelo de la aplicación
    private SobreAppFragment fragment;

    /**
     * Constructor principal de la clase
     *
     * @param fragment Fragmento (Vista) con la que mantiene una relación 1-a-1.
     * @param model    Modelo (único) de la aplicación.
     */
    public SobreAppPresenter(Fragment fragment, MainModel model) {
        this.fragment = (SobreAppFragment) fragment;
        this.model = model;
    }

    /**
     * Método que realiza las operaciones iniciales cuando se crea el
     * <tt>Fragment SobreAppFragment</tt>.
     *
     * <p>Se encarga de marcar el quinto ítem del <tt>NavigationView</tt> (correspondiente a
     * <tt>SobreAppFragment</tt>), fijar la orientación de la pantalla en "Portrait",
     * utilizar el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>, y, a partir del icono
     * principal de la aplicación (imagen con formato rectangular), transformarlo en una imagen con
     * un borde circular y enviárselo a la Vista (<tt>SobreAppFragment</tt>) para que se lo asigne
     * al objeto <tt>ImageView</tt> que contiene el icono de la aplicación, mediante el uso de la
     * clase <tt>RoundedBitmapDrawable</tt>.</p>
     *
     * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
     */
    @Override
    public void setUpInitialSettings(Toolbar toolbar) {
        // Marcar el quinto ítem del Navigationview
        ((MainActivity) fragment.getActivity()).setNavigationItem(4);

        // Fijar la orientación de la pantalla en "Portrait"
        fragment.getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);

        // Utilizar el toolbar como SupportActionBar
        AppCompatActivity activity = ((AppCompatActivity) fragment.getActivity());

        activity.setSupportActionBar(toolbar);

        if (activity.getSupportActionBar() != null)
            activity.getSupportActionBar().setHomeAsUpIndicator(R.mipmap.ic_menu);

        activity.getSupportActionBar().setDisplayHomeAsUpEnabled(true);
        activity.getSupportActionBar().setTitle("");

        // Transformar el icono original de la aplicación, aplicando un borde alrededor del mismo
        // Get the Resources
        Resources resources = fragment.getResources();

        // Get the bitmap from drawable resources
        Bitmap bitmap = BitmapFactory.decodeResource(resources, R.drawable.ic_launcher);

        // Create a new RoundedBitmapDrawable
        RoundedBitmapDrawable drawable = RoundedBitmapDrawableFactory.create(resources, bitmap);

        // Calculate the bitmap radius and apply it to the RoundedBitmapDrawable
        drawable.setCornerRadius(Math.min(drawable.getMinimumWidth(), drawable.getMinimumHeight())
                / 2.0f);

        // Enviar el objeto RoundedBitMapDrawable a la Vista para que se lo asigne al ImageView
        fragment.setImageDrawable(drawable);
    }

    /**
     * Perform any final cleanup before an activity is destroyed.
     * <p>
     * Se encarga de liberar la referencia al <tt>Fragment</tt> con el que está asociado.
     */
    @Override
    public void onDestroy() {
        fragment = null;
    }
}
