package com.uva.rafael.tfg_goniometer.view.adapters;

import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.TextView;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.presenter.UltimasMedicionesPresenter;

import java.util.ArrayList;

/**
 * Este es el <tt>RecyclerView.Adapter</tt> asociado al <tt>RecyclerView</tt> de
 * <tt>UltimasMedicionesFragment</tt>.
 *
 * <p>En concreto, se encarga de mostrar toda la información relativa a las últimas 10
 * mediciones que ha almacenado el usuario en la Base de Datos, y notificar a
 * <tt>UltimasMedicionesPresenter</tt> cuando el usuario pulse sobre el botón para borrar la medición
 * en alguna de ellas.</p>
 *
 * @author Rafael Matamoros Luque
 * @see UltimasMedicionesPresenter
 * @see android.support.v7.widget.RecyclerView.ViewHolder
 * @version 1.0
 */

public class MedicionListAdapter extends RecyclerView.Adapter<MedicionListAdapter.ViewHolder> {

    private final ArrayList<ArrayList<String>> medicionesList; // Listado de mediciones a mostrar
    // Presentador asociado con este RecyclerView.Adapter
    private final UltimasMedicionesPresenter presenter;

    /**
     * Constructor principal del <tt>RecyclerView.Adapter</tt> para el <tt>RecyclerView</tt>
     *
     * @param medicionesList Listado de mediciones a mostrar
     * @param presenter      Presentador asociado con este <tt>RecyclerView.Adapter</tt>
     */
    public MedicionListAdapter(ArrayList<ArrayList<String>> medicionesList,
                               UltimasMedicionesPresenter presenter) {
        this.medicionesList = medicionesList;
        this.presenter = presenter;
    }

    /**
     * Método llamado para instanciar cada uno de los elementos del <tt>RecyclerView.Adapter</tt>
     * con su layout asociada (R.layout.medicion_miniatura, en este caso).
     *
     * @param parent   The ViewGroup into which the new View will be added after it is bound to
     *                 an adapter position.
     * @param viewType The view type of the new View.
     * @return A new ViewHolder that holds a View of the given view type.
     *
     * @see #getItemViewType(int)
     * @see #onBindViewHolder(ViewHolder, int)
     */
    @Override
    public ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View vista = LayoutInflater.from(parent.getContext()).inflate(R.layout.medicion_miniatura, parent, false);

        return new ViewHolder(vista);
    }

    /**
     * Called by RecyclerView to display the data at the specified position. This method should
     * update the contents of the {@link ViewHolder#itemView} to reflect the item at the given
     * position.
     *
     * <p>En concreto, se encarga de mostrar toda la información relativa a las últimas 10
     * mediciones que ha almacenado el usuario en la Base de Datos, además de añadir un evento de
     * tipo <tt>OnClickListener</tt> al botón para borrar la medición, para notificar al Presentador
     * cuando el usuario lo pulse.</p>
     *
     * @param holder   The ViewHolder which should be updated to represent the contents of the
     *                 item at the given position in the data set.
     * @param position The position of the item within the adapter's data set.
     */
    @Override
    public void onBindViewHolder(final ViewHolder holder, int position) {
        holder.fechaHora.setText(medicionesList.get(position).get(0));
        holder.medicion.setText(medicionesList.get(position).get(1));
        holder.articulacion.setText(medicionesList.get(position).get(2));
        holder.movimiento.setText(medicionesList.get(position).get(3));
        holder.nombrePaciente.setText(medicionesList.get(position).get(4)
                + " - ID " + medicionesList.get(position).get(5));

        holder.borrarMedicion.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                presenter.onDeleteMeasurementClicked(holder.nombrePaciente.getText().toString(),
                        (holder.fechaHora.getText().toString()));
            }
        });
    }

    /**
     * Returns the total number of items in the data set held by the adapter.
     *
     * @return The total number of items in this adapter.
     */
    @Override
    public int getItemCount() {
        if (medicionesList != null)
            return medicionesList.size();
        return 0;
    }

    /*
     * A ViewHolder describes an item view and metadata about its place within the RecyclerView.
     */
    static class ViewHolder extends RecyclerView.ViewHolder {

        // Elementos del CardView
        private final TextView nombrePaciente, fechaHora, articulacion, movimiento, medicion;
        private final Button borrarMedicion;

        // Constructor de la clase. Los elementos del CardView son bindeados
        private ViewHolder(View vista) {
            super(vista);

            nombrePaciente = (TextView) vista.findViewById(R.id.paciente_textView);
            fechaHora = (TextView) vista.findViewById(R.id.fechaHora);
            articulacion = (TextView) vista.findViewById(R.id.articulacion);
            movimiento = (TextView) vista.findViewById(R.id.movimiento);
            medicion = (TextView) vista.findViewById(R.id.lectura_goniometro);
            borrarMedicion = (Button) vista.findViewById(R.id.borrar_medicion);
        }
    }
}
