package com.uva.rafael.tfg_goniometer.view.adapters;

import android.os.Bundle;
import android.support.v7.widget.RecyclerView;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.presenter.PacientesPresenter;

import java.util.ArrayList;

/**
 * Este es el <tt>RecyclerView.Adapter</tt> asociado al <tt>RecyclerView</tt> de
 * <tt>PacientesFragment</tt>.
 *
 * <p>En concreto, se encarga de mostrar el nombre de cada uno de los pacientes almacenados
 * en la Base de Datos, que coincidan con los criterios de filtrado que se hayan introducido (si
 * es que se ha introducido alguno), y notificar a <tt>PacientesPresenter</tt> cuando el usuario
 * pulse sobre alguno de ellos.</p>
 *
 * @author Rafael Matamoros Luque
 * @see PacientesPresenter
 * @see android.support.v7.widget.RecyclerView.ViewHolder
 * @version 1.0
 */
public class PacienteListAdapter extends RecyclerView.Adapter<PacienteListAdapter.ViewHolder> {

    private final ArrayList<ArrayList<String>> pacientesList; // Listado de pacientes a mostrar
    // Bundle que se envía al Presentador para la creación de otros Fragments
    private final Bundle args;
    // Presentador asociado con este RecyclerView.Adapter
    private final PacientesPresenter presenter;

    /**
     * Constructor principal del <tt>RecyclerView.Adapter</tt> para el <tt>RecyclerView</tt>
     *
     * @param pacienteList Listado de pacientes a mostrar
     * @param args Parámetros en formato <tt>Bundle</tt>
     * @param presenter Presentador asociado con este <tt>RecyclerView.Adapter</tt>
     */
    public PacienteListAdapter(ArrayList<ArrayList<String>> pacienteList, Bundle args,
                               PacientesPresenter presenter) {
        this.pacientesList = pacienteList;
        this.args = args;
        this.presenter = presenter;
    }

    /**
     * Método llamado para instanciar cada uno de los elementos del <tt>RecyclerView.Adapter</tt>
     * con su layout asociada (R.layout.paciente_miniatura, en este caso).
     *
     * @param parent   The ViewGroup into which the new View will be added after it is bound to
     *                 an adapter position.
     * @param viewType The view type of the new View.
     * @return A new ViewHolder that holds a View of the given view type.
     *
     * @see #getItemViewType(int)
     * @see #onBindViewHolder(ViewHolder, int)
     */
    @Override
    public ViewHolder onCreateViewHolder(ViewGroup parent, int viewType) {
        View vista = LayoutInflater.from(parent.getContext()).inflate(R.layout.paciente_miniatura, parent, false);

        return new ViewHolder(vista);
    }

    /**
     * Called by RecyclerView to display the data at the specified position. This method
     * should update the contents of the {@link ViewHolder#itemView} to reflect the item at
     * the given position.
     *
     * <p>En concreto, se encarga de mostrar el nombre de cada uno de los pacientes almacenados
     * en la Base de Datos, que coincidan con los criterios de filtrado que se hayan introducido (si
     * es que se ha introducido alguno) y añadir a cada uno un evento de tipo <tt>OnClickListener</tt>,
     * para notificar al Presentador cuando el usuario pulse sobre alguno de ellos.</p>
     *
     * @param holder   The ViewHolder which should be updated to represent the contents of the
     *                 item at the given position in the data set.
     * @param position The position of the item within the adapter's data set.
     */
    @Override
    public void onBindViewHolder(ViewHolder holder, int position) {
        final String nombrePaciente = pacientesList.get(position).get(0);
        final int idPaciente = Integer.parseInt(pacientesList.get(position).get(1));

        if (nombrePaciente != null)
            holder.nombrePaciente.setText(nombrePaciente);

        holder.pacienteLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                presenter.onLayoutClicked(nombrePaciente, idPaciente, args);
            }
        });
    }

    /**
     * Returns the total number of items in the data set held by the adapter.
     *
     * @return The total number of items in this adapter.
     */
    @Override
    public int getItemCount() {
        if (pacientesList != null)
            return pacientesList.size();
        return 0;
    }

    /*
     * A ViewHolder describes an item view and metadata about its place within the RecyclerView.
     */
    static class ViewHolder extends RecyclerView.ViewHolder {

        // Elementos del CardView
        private final TextView nombrePaciente;
        private final LinearLayout pacienteLayout;

        // Constructor de la clase. Los elementos del CardView son bindeados
        private ViewHolder(View vista) {
            super(vista);

            nombrePaciente = (TextView) vista.findViewById(R.id.nombrePaciente);
            pacienteLayout = (LinearLayout) vista.findViewById(R.id.paciente_cardview);
        }
    }
}
