package com.uva.rafael.tfg_goniometer.data.entities;

import android.content.ContentValues;
import android.text.TextUtils;

import com.uva.rafael.tfg_goniometer.data.Contract;
import com.uva.rafael.tfg_goniometer.data.enums.Sexo;

import java.util.ArrayList;

/**
 * Esta clase modela a la <tt>Entity</tt> "Paciente" de la Base de Datos.
 *
 * <p>Se encarga de proveer un método para insertar un "Paciente" en la Base de Datos, pasando cada
 * uno de los atributos que componen esta clase a un formato adecuado, por medio de un objeto de
 * tipo <tt>ContentValues</tt>.</p>
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see ArrayList
 * @see ContentValues
 * @see com.uva.rafael.tfg_goniometer.data.Contract.PatientEntry
 * @version 1.0
 */

public class Paciente {
    private final String nombre; // Nombre del paciente
    private final int edad; // Edad del paciente
    private final Sexo sexo; // Sexo del paciente
    private final int id; // ID del paciente
    private final String diagnostico; // Diagnóstico del paciente
    private final ArrayList<String> tags; // Tags para identificar mas facilmente al paciente

    private final long telefono; // Teléfono del paciente
    private final String direccion; // Dirección del paciente
    private final String sintomas; // Síntomas del paciente
    private final String tratamiento_previo; // Tratamiento previo que ha estado siguiendo el paciente
    private final String tratamiento_actual; // Tratamiento actual que esta siguiendo el paciente
    private final String comentarios; // Comentarios adicionales

    /**
     * Constructor principal de la clase
     *
     * @param nombre             Nombre
     * @param edad               Edad
     * @param sexo               Sexo
     * @param diagnostico        Diagnóstico
     * @param tags               Tags para identificar mas facilmente al paciente
     * @param id                 ID
     * @param telefono           Teléfono
     * @param direccion          Dirección
     * @param sintomas           Síntomas
     * @param tratamiento_previo Tratamiento previo que ha estado siguiendo el paciente
     * @param tratamiento_actual Tratamiento actual que esta siguiendo el paciente
     * @param comentarios        Comentarios adicionales
     */
    public Paciente(String nombre, int edad, Sexo sexo, int id, String diagnostico,
                    ArrayList<String> tags, long telefono, String direccion, String sintomas,
                    String tratamiento_previo, String tratamiento_actual, String comentarios) {
        this.nombre = nombre;
        this.edad = edad;
        this.sexo = sexo;
        this.id = id;
        this.diagnostico = diagnostico;
        this.tags = tags;
        this.telefono = telefono;
        this.direccion = direccion;
        this.sintomas = sintomas;
        this.tratamiento_previo = tratamiento_previo;
        this.tratamiento_actual = tratamiento_actual;
        this.comentarios = comentarios;
    }

    /**
     * Método que se encarga de dar un formato adecuado a cada uno de los atributos de la clase,
     * para, posteriormente, ser insertados en la Base de Datos.
     *
     * @return values <tt>ContentValues</tt> con los atributos del Paciente ya formateados
     */
    public ContentValues toContentValues() {
        // ContentValues a devolver con los atributos de la medición ya formateados
        ContentValues values = new ContentValues();

        // Relacionar el campo de la tabla en la BD con el atributo que corresponde de la clase
        values.put(Contract.PatientEntry.NOMBRE, nombre);
        values.put(Contract.PatientEntry.EDAD, edad);
        values.put(Contract.PatientEntry.SEXO, sexo.toString());
        values.put(Contract.PatientEntry.ID, id);
        values.put(Contract.PatientEntry.DIAGNOSTICO, diagnostico);
        values.put(Contract.PatientEntry.TAGS, arrayListToString(tags));
        values.put(Contract.PatientEntry.TELEFONO, telefono);
        values.put(Contract.PatientEntry.DIRECCION, direccion);
        values.put(Contract.PatientEntry.SINTOMAS, sintomas);
        values.put(Contract.PatientEntry.TRATAMIENTO_PREVIO, tratamiento_previo);
        values.put(Contract.PatientEntry.TRATAMIENTO_ACTUAL, tratamiento_actual);
        values.put(Contract.PatientEntry.COMENTARIOS, comentarios);

        // Devolver el ContentValues con los datos en el formato correspondiente
        return values;
    }

    /*
     * Método que se encarga de, dado un ArrayList<String>, devolver un único String formado por
     * cada uno de los componentes del ArrayList, separados por ','
     */
    private String arrayListToString(ArrayList<String> tags) {
        // Para realizar esta tarea, se utiliza el método "join" de la clase "TextUtils"
        return TextUtils.join(", ", tags);
    }
}
