package com.uva.rafael.tfg_goniometer.view.fragments;

import android.app.Fragment;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.Toolbar;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;
import android.widget.Toast;

import com.uva.rafael.tfg_goniometer.R;
import com.uva.rafael.tfg_goniometer.dependencyInjection.App;
import com.uva.rafael.tfg_goniometer.interfaces.ViewFunctions;
import com.uva.rafael.tfg_goniometer.presenter.UltimasMedicionesPresenter;

import javax.inject.Inject;

/**
 * Esta clase es el <tt>Fragment</tt> que se encarga de mostrar el listado de las últimas 10
 * mediciones almacenadas por el usuario en la Base de Datos, ordenadas de más a menos recientes.
 *
 * <p>Esta clase forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informática (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see UltimasMedicionesPresenter
 * @see Toolbar
 * @see RecyclerView
 * @version 1.0
 */
public class UltimasMedicionesFragment extends Fragment implements ViewFunctions,
        ViewFunctions.UltimasMedicionesFunctions {

    /*
     * Presentador asociado con este Fragment (Vista). Su instancia se obtiene mediante Inversión
     * de Dependencias
     */
    @SuppressWarnings({"unused", "CanBeFinal"})
    @Inject
    UltimasMedicionesPresenter presenter;

    /*
     * RecyclerView que se emplea para mostrar el listado de las últimas 10 mediciones
     * almcenadas por el usuario
     */
    private RecyclerView recyclerView;

    /**
     * Método llamado para instanciar el <tt>Fragment</tt> con su layout asociada
     * (R.layout.ultimas_mediciones_fragment, en este caso).
     *
     * @param inflater           The LayoutInflater object that can be used to inflate any view in the
     *                           fragment.
     * @param container          If non-null, this is the parent view that the fragment's UI should be
     *                           attached to. The fragment should not add the view itself, but this can be
     *                           used to generate the LayoutParams of the view.
     * @param savedInstanceState If non-null, this fragment is being re-constructed from a previous
     *                           saved state as given here. Return the View for the fragment's UI,
     *                           or null.
     */
    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container,
                             Bundle savedInstanceState) {

        // Obtención de la instancia de UltimasMedicionesPresenter
        ((App) getActivity().getApplication()).getComponent(this).injectUltimasMedicionesFragment(this);

        // Inflar el layout del Fragment
        View vista = inflater.inflate(R.layout.ultimas_mediciones_fragment, container, false);

        // Bindeo del Toolbar
        Toolbar toolbar = (Toolbar) vista.findViewById(R.id.toolbar);

        // Configuración inicial
        presenter.setUpInitialSettings(toolbar);

        /*
         * Bindeo del RecyclerView donde se muestra el listado de las últimas 10 mediciones
         * almacenadas, y configuración del mismo
         */
        recyclerView = (RecyclerView) vista.findViewById(R.id.recyler_view);
        presenter.configRecyclerView(recyclerView);

        return vista;
    }

    /**
     * Método que muestra un mensaje de éxito cuando se ha logrado eliminar correctamente una
     * medición de la Base de Datos.
     */
    @Override
    public void displaySuccess() {
        Toast toast = Toast.makeText(getActivity(), R.string.delete_measurement_success,
                Toast.LENGTH_LONG);

        TextView textView = (TextView) toast.getView().findViewById(android.R.id.message);

        if (textView != null)
            textView.setGravity(Gravity.CENTER); // Centrar el mensaje en el Toast

        toast.show();
    }

    /**
     * Método que muestra un mensaje de error cuando se ha intentado eliminar una medición de la
     * Base de Datos.
     */
    @Override
    public void displayError() {
        Toast toast = Toast.makeText(getActivity(), R.string.delete_measurement_error,
                Toast.LENGTH_LONG);

        TextView textView = (TextView) toast.getView().findViewById(android.R.id.message);

        if (textView != null)
            textView.setGravity(Gravity.CENTER); // Centrar el mensaje en el Toast

        toast.show();
    }

    /**
     * Método que simplemente muestra el <tt>AlertDialog</tt> que ha recibido como parámetro.
     *
     * @param dialog <tt>AlertDialog</tt> a mostrar
     */
    @Override
    public void displayResult(AlertDialog dialog) {
        dialog.show();
    }

    /**
     * Método que se encarga de re-asignar el <tt>RecyclerView.Adapter</tt>
     * (<tt>MedicionListAdapter</tt>, en este caso) al <tt>RecyclerView</tt> del <tt>Fragment</tt>.
     *
     * @param adapter <tt>RecyclerView.Adapter</tt> a asignar
     */
    @Override
    public void setAdapter(RecyclerView.Adapter adapter) {
        recyclerView.setAdapter(adapter);
    }

    /**
     * Perform any final cleanup before an activity is destroyed.
     */
    @Override
    public void onDestroy() {
        presenter.onDestroy();
        super.onDestroy();
    }
}
