package com.uva.rafael.tfg_goniometer.interfaces;

import java.util.ArrayList;

/**
 * Esta interfaz engloba los métodos públicos de la clase <tt>MainModel</tt> relacionados con la
 * gestión de la Base de Datos a través de la clase de tipo <tt>SQLiteHelper</tt> -
 * <tt>GoniometerDBHelper</tt>.
 *
 * <p>Esta interfaz forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see com.uva.rafael.tfg_goniometer.model.MainModel
 * @version 1.0
 */
@SuppressWarnings("unused")
public interface ModelFunctions {

    /**
     * Método que realiza una consulta a la Base de Datos para almacenar en la misma la información
     * del paciente que recibe como parámetro
     *
     * @param nuevoPaciente <tt>ArrayList<String></tt> a convertir en <tt>Paciente</tt> y almacenar
     *                      en la Base de Datos
     * @return long > 0 si la consulta se realizó con éxito, -1 en caso contrario
     */
    long storePatient(ArrayList<String> nuevoPaciente);

    /**
     * Método que, dado el <tt>ArrayList<String></tt> con toda la información introducida por el
     * usuario sobre la medición, se encarga de obtener una instancia de tipo <tt>Medicion</tt> y
     * realizar una consulta a la Base de Datos para almacenarla.
     *
     * @param nuevaMedicion <tt>ArrayList<String></tt> a convertir en <tt>Medicion</tt> y almacenar
     *                      en la Base de Datos
     * @return long > 0 si la consulta se realizó con éxito, -1 en caso contrario
     */
    long storeMeasurement(ArrayList<String> nuevaMedicion);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos y devolver una lista con
     * todos los nombres e ids de los pacientes almacenados en la misma.
     *
     * @return pacientes <tt>ArrayList<ArrayList<String>></tt> con los nombres e ids de todos los
     * pacientes almacenados en la Base de Datos
     */
    ArrayList<ArrayList<String>> getPacientes();

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos y devolver una lista con
     * la información a mostrar de un <tt>Paciente</tt> (en concreto, edad, sexo, diagnóstico,
     * comentarios adicionales y tags) a partir del nombre y el ID del mismo.
     *
     * @param nombre Nombre del paciente del que recuperar la información
     * @param id ID del paciente del que recuperar la información
     * @return informacionPaciente <tt>ArrayList<String></tt> con toda la información del paciente
     *                              almacenada en la Base de Datos
     */
    ArrayList<String> getPacienteBy(String nombre, int id);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos y devolver una lista con
     * todas las mediciones que se han realizado a un <tt>Paciente</tt> en concreto, a partir del
     * nombre e ID del mismo.
     *
     * @param nombre Nombre del paciente del que recuperar la información
     * @param id ID del paciente del que recuperar la información
     * @return medicionesPaciente <tt>ArrayList<ArrayList<String>></tt> con todas las mediciones
     *                              que se han realizado al paciente
     */
    ArrayList<ArrayList<String>> getMedicionesPacienteBy(String nombre, int id);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos para, primero, borrar todas
     * las mediciones que se han realizado a un paciente, y, después, eliminar al paciente en
     * cuestión de la misma, a partir del nombre e ID del paciente.
     *
     * @param nombre Nombre del paciente del que eliminar toda la información
     * @param id ID del paciente del que eliminar toda la información
     * @return int == 1 si el borrado se realizó con éxito, 0 en caso contrario
     */
    int onDeletePatient(String nombre, int id);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos y devolver un listado de
     * los nombres e IDs de los pacientes que se encuentran en la misma, filtrados según el criterio
     * seleccionado por el usuario en el <tt>Spinner</tt> de <tt>PacientesFragment</tt> y la
     * información introducida en el <tt>EditText</tt> del mismo <tt>Fragment</tt>.
     *
     * @param texto Información introducida por el usuario en el <tt>EditText</tt>
     * @param filtro Filtro seleccionado por el usuario en el <tt>Spinner</tt>
     * @return listadoPacientesFiltrado <tt>ArrayList<ArrayList<String>></tt> con los nombres de
     *                                  todos los pacientes que coincidan con los criterios de
     *                                  filtración introducidos por el usuario
     */
    ArrayList<ArrayList<String>> getPacientesFiltrado(String texto, String filtro);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos y devolver una lista con
     * las últimas 10 mediciones (como maximo) que se han almacenado en la misma, ordenadas de mas
     * a menos recientes.
     *
     * @return mediciones <tt>ArrayList<ArrayList<String>></tt> con las últimas 10 mediciones (como
     *                    maximo) que se han almacenado en la Base de Datos
     */
    ArrayList<ArrayList<String>> getUltimasMediciones();

    /**
     * Método que se encarga de, a partir de los datos correspondientes a una medición, dar un
     * formato adecuado a los mismos para que se puedan mostrar correctamente en la pantalla del
     * dispositivo.
     *
     * <p>Principalmente realiza tareas de añadir saltos de línea entre cada palabra (con el fin de
     * evitar que alguna parte de la medición no se pueda visualizar en el dispositivo), así como de
     * dejar sólo la primera letra de cada palabra con mayúscula, y pasando el resto de la misma a
     * minúscula, para que así el texto ocupe menos espacio (los datos, por defecto, se obtienen de
     * la Base de Datos con todos su caracteres en mayúscula).</p>
     *
     * <p>Por último, este método tiene en cuenta el idioma del dispositivo para mostrar la
     * información de un modo u otro, atendiendo a criterios sintacticos.</p>
     *
     * @param medicion <tt>ArrayList<String></tt> sobre la que aplicar las correciones de formato
     * @return <tt>ArrayList<String></tt> con todos los campos de la medición con el correspondiente
     *         formato ya aplicado para su correcta visualización en la pantalla del dispositivo
     */
    ArrayList<String> getDatosFormateados(ArrayList<String> medicion);

    /**
     * Método que se encarga de realizar una consulta a la Base de Datos para borrar una medición
     * de la misma, a partir de la fecha-hora en la que se realizó.
     *
     * @param fechaHora Fecha y hora en la que se realizó la medición
     * @return int == 1 si el borrado se realizó con éxito, 0 en caso contrario
     */
    int onDeleteMeasurement(String fechaHora);
}
