package com.uva.rafael.tfg_goniometer.interfaces;

import android.os.Bundle;
import android.support.design.widget.NavigationView;
import android.support.v4.widget.DrawerLayout;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.Toolbar;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.Spinner;

/**
 * Esta interfaz engloba todos los métodos públicos de las clases de tipo <tt>Presenter</tt> que se
 * encuentran en el paquete "presenter" de esta aplicación.
 *
 * <p>Esta interfaz posee un único metodo <tt>onDestroy()</tt> común a todas las interfaces que
 * posee. Ademas de esto, para cada uno de los Presentadores, existe una interfaz que engloba los
 * métodos específicos de cada uno de ellos.</p>
 *
 * <p>Esta interfaz forma parte de la aplicación TFG-Goniometer, desarrollada para el Trabajo de
 * Fin de Grado - Grado en Ingeniería Informatica (Universidad de Valladolid)</p>
 *
 * @author Rafael Matamoros Luque
 * @see NavigationView
 * @see DrawerLayout
 * @see Menu
 * @see MenuItem
 * @see Toolbar
 * @see Spinner
 * @see RecyclerView
 * @version 1.0
 */
@SuppressWarnings("unused")
public interface PresenterFunctions {

    /**
     * Perform any final cleanup before an activity is destroyed.
     * <p>
     * Se encarga de liberar la referencia a la <tt>Activity</tt> o <tt>Fragment</tt> con el que
     * esta asociado.
     */
    void onDestroy();

    /**
     * Interfaz correspondiente al Presentador <tt>HomePresenter</tt>
     */
    interface HomeFunctions {

        /**
         * Método que realiza las operaciones iniciales cuando se crea el
         * <tt>Fragment HomeFragment</tt>.
         * <p>
         * Se encarga de marcar el primer ítem del <tt>NavigationView</tt> (correspondiente a
         * <tt>HomeFragment</tt>), fijar la orientación de la pantalla en "Portrait" y de utilizar
         * el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>.
         *
         * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
         */
        void setUpInitialSettings(Toolbar toolbar);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "NUEVA MEDICIÓN" en el <tt>HomeFragment</tt>.
         * <p>
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>MedicionFragment</tt>.
         */
        void onNuevaMedicionClicked();

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "ULTIMAS 10 MEDICIONES" en el <tt>HomeFragment</tt>.
         * <p>
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>UltimasMedicionesFragment</tt>.
         */
        void onUltimasMedicionesClicked();

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "PERFILES DE PACIENTES" en el <tt>HomeFragment</tt>.
         * <p>
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>PacientesFragment</tt>.
         */
        void onPerfilPacientesClicked();
    }

    /**
     * Interfaz correspondiente al Presentador <tt>MainPresenter</tt>
     */
    interface MainFunctions {

        /**
         * Método que realiza las operaciones iniciales cuando se crea la actividad
         * <tt>MainActivity</tt>.
         *
         * Crea el <tt>Fragment</tt> inicial de la aplicación (<tt>HomeFragment</tt>) y lo añade
         * al contenedor de <tt>Fragments</tt>. Ademas, proporciona el <tt>Context</tt> a la clase
         * <tt>SQLiteHelper</tt> que sirve de conexión con la BD (<tt>GoniometerDBHelper</tt>).
         */
        void setUpInitialSettings();

        /**
         * Método que obtiene las referencias al <tt>NavigationView</tt> y <tt>DrawerLayout</tt> de
         * <tt>MainActivity</tt>. Ademas asigna un evento de tipo <tt>NavigationItemListener</tt> al
         * <tt>NavigationView</tt> y crea los respectivos <tt>Fragments</tt> cuando se pulsa sobre
         * uno de los elementos del mismo.
         *
         * @param navigationView Menú sobre el que se aplicara el evento
         *                       <tt>NavigationItemListener</tt>
         * @param drawerLayout   DrawerLayout que contiene al <tt>NavigationView</tt>
         */
        void setNavigationItemSelectedListener(NavigationView navigationView,
                                               DrawerLayout drawerLayout);

        /**
         * Initialize the contents of the Activity's standard options menu.
         *
         * @param menu The options menu in which we place our items.
         */
        void onCreateOptionsMenu(Menu menu);

        /**
         * Prepare the Screen's standard options menu to be displayed
         *
         * @param menu The options menu as last shown or first initialized by
         *             <tt>onCreateOptionsMenu()</tt>.
         */
        void onPrepareOptionsMenu(Menu menu);

        /**
         * This hook is called whenever an item in your options menu is selected.
         *
         * @param item The menu item that was selected.
         * @return Return false to allow normal menu processing to proceed, true to consume it here.
         */
        boolean onOptionsItemSelected(MenuItem item);

        /**
         * Método que se emplea para marcar como seleccionado uno de los distintos items en el
         * <tt>NavigationView</tt>.
         *
         * @param code Posición del item a marcar dentro del <tt>NavigationView</tt>
         */
        void setNavigationItem(int code);

        /**
         * Called when the activity (<tt>MainActivity</tt>) has detected the user's press of the
         * back key.
         */
        void onBackPressed();
    }

    /**
     * Interfaz correspondiente al Presentador <tt>MedicionPresenter</tt>
     */
    interface MedicionFunctions {

        /**
         * Método que realiza las operaciones iniciales cuando se crea el
         * <tt>Fragment MedicionFragment</tt>.
         * <p>
         * Se encarga de marcar el segundo ítem del <tt>NavigationView</tt> (correspondiente a
         * <tt>MedicionFragment</tt>), fijar la orientación de la pantalla en "Landscape", de utilizar
         * el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>, y de registrar el
         * <tt>Listener</tt> del vector de rotación, para que empiece a obtener lecturas del dispositivo.
         * <p>
         * <p>Ademas de esto, se encarga de enviar a la Vista (<tt>MedicionFragment</tt>), el tamaño
         * que tiene que asignar al goniómetro y al botón "Reiniciar" en función del tamaño de la
         * pantalla del dispositivo.</p>
         *
         * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
         */
        void setUpInitialSettings(Toolbar toolbar);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el
         * <tt>LinearLayout</tt> que contiene el goniómetro en el <tt>MedicionFragment</tt>.
         * <p>
         * Se encarga de comprobar si se dispone un lectura del vector de rotación como "origen" para
         * calcular el angulo de desplazamiento.
         * <p>
         * Si ya se dispone de ella, significa que el usuario ha pulsado sobre el goniómetro para terminar
         * la medición, por tanto, se deja de mostrar la variación del angulo en tiempo real, y se
         * obtiene la fecha y la hora en la que se ha realizado la medición para su, quizas, posterior
         * almacenamiento.
         * <p>
         * Si no se dispone de ella, significa que el usuario ha pulsado sobre el goniómetro para empezar
         * la medición, por tanto, se toma como origen de la misma la lectura que se esta obteniendo
         * del vector de rotación en esa posición, y se empieza a mostrar la variación del angulo en
         * tiempo real.
         */
        void onGoniometerClicked();

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "ALMACENAR MEDICIÓN" en el <tt>MedicionFragment</tt>.
         *
         * Se encarga de:
         *
         * 1.- Comprobar que el usuario ha realizado una medición y no ha pulsado sobre el botón
         * antes de haberlo hecho.
         * 2.- Crea un dialogo que envía a la Vista para que se lo muestre al usuario, con el fin de
         * confirmar que quiere almacenar la medición.
         * 3.- Crea un nuevo <tt>Fragment PacientesFragment</tt> para que el usuario elija el
         * usuario al que quiere añadir esa medición.
         *
         * @param measurement Lectura obtenida del <tt>TextView</tt> con el angulo de desplazamiento
         *                    del dispositivo.
         */
        void onStoreMeasurementClicked(String measurement);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "Reiniciar" en el <tt>MedicionFragment</tt>.
         *
         * Se encarga de:
         *
         * 1.- Comprobar que la lectura del goniómetro en ese momento es distinta de "0.0º".
         * 2.- De ser así, indica que se deje de mostrar la variación del angulo en tiempo real.
         * 3.- Reinicia los valores que representan el origen de la medición a 0.
         * 4.- Vuelve al estado inicial la lectura del goniómetro, así como la barra de
         * desplazamiento que representa la variación del angulo.
         *
         * @param lecturaGoniometro Lectura del goniómetro que se tiene en ese momento.
         */
        void onResetClicked(String lecturaGoniometro);


        /**
         * Called as part of the fragment lifecycle when your fragment is ready to start interacting
         * with the user.
         * <p>
         * Se encarga de volver a registrar el vector de rotación para poder obtener, de nuevo, lecturas
         * del mismo.
         */
        void onResume();

        /**
         * Called as part of the fragment lifecycle when a fragment is going into the background, but
         * has not (yet) been killed.
         * <p>
         * Se encarga de borrar el <tt>Listener</tt> del vector de rotación, para evitar así, que se
         * consuma la batería del dispositivo cuando la actividad esté en un segundo plano.
         */
        void onPause();
    }

    /**
     * Interfaz correspondiente al Presentador <tt>NuevaMedicionPresenter</tt>
     */
    interface NuevaMedicionFunctions {

        /**
         * Método que se encarga de enviar a la Vista (<tt>NuevaMedicionFragment</tt>) los nombres
         * de las constantes para obtener los parámetros de tipo <tt>Bundle</tt> recibidos en la
         * creación del <tt>Fragment</tt>.
         */
        void onCreate();

        /**
         * Método que se encarga de "cargar" con contenido cada uno de los <tt>Spinners</tt>
         * inicialmente.
         * <p>
         * Sólo se "cargan" por completo los <tt>Spinners</tt> que indican el lado del cuerpo en el
         * que se hace la medición, y el que indica si el movimiento ha sido realizado única y
         * exclusivamente por el paciente, o ha necesitado ayuda para realizarlo.
         * <p>
         * El resto de <tt>Spinners</tt> se "cargan" con un valor por defecto para que no aparezan
         * vacíos en un principio.
         *
         * @param jointSide    <tt>Spinner</tt> en el que se indica el lado del cuerpo en el que se
         *                      encuentra la articulación sobre la que se ha realizado la medición
         * @param joint         <tt>Spinner</tt> en el que se indica la articulación en concreto que
         *                      ha realizado el movimiento
         * @param movement      <tt>Spinner</tt> en el que se indica el movimiento que ha realizado
         *                      dicha articulación
         * @param movementType <tt>Spinner</tt> en el que se indica si el movimiento ha sido
         *                     realizado única y exclusivamente por el usuario, o ha necesitado
         *                     ayuda
         */
        void loadSpinners(Spinner jointSide, Spinner joint, Spinner movement, Spinner movementType);

        /**
         * Método que se encarga de obtener la información relativa a un paciente a partir del
         * nombre e ID del mismo, mediante una consulta a la Base de Datos.
         *
         * Posteriormente, envía esta información a la Vista (<tt>NuevaMedicionFragment</tt>) para
         * que se la muestre al usuario.
         *
         * @param nombre Nombre del paciente
         * @param id ID del paciente
         * @param fechaHora Fecha y hora a la que se realizó la medición
         * @param lecturaGoniometro Lectura final del goniómetro (angulo de desplazamiento)
         */
        void getInformacionPaciente(String nombre, int id, String fechaHora, String lecturaGoniometro);

        /**
         * Evento <tt>ItemSelectedListener</tt> que se llama cuando el usuario ha pulsado sobre
         * alguno de los elementos de uno de los <tt>Spinner</tt> en <tt>NuevaMedicionFragment</tt>.
         *
         * <p>Se encarga de, en función de cual de los 4 <tt>Spinner</tt> es el que recibe el evento,
         * "cargar" el siguiente <tt>Spinner</tt> con un determinado <tt>Array</tt> en función del
         * ítem que se haya seleccionado sobre el primero para, así, limitar la articulación y el
         * movimiento que puede introducir el usuario, evitando, de esta manera, introducir valores
         * absurdos en articulaciones que no pueden realizar un determinado movimiento.
         *
         * Ademas, se encarga de mostrar u ocultar el <tt>LinearLayout</tt> que contiene el
         * <tt>EditText</tt> en el que el usuario puede escribir, en caso de haber seleccionado en
         * alguno de los <tt>Spinner</tt> la opción "Otro/a".</p>
         *
         * @param spinner1 <tt>Spinner</tt> que recibe el evento <tt>ItemSelected</tt>
         * @param spinner2 <tt>Spinner</tt> que, opcionalmente, se "cargara" en función del valor
         *                 seleccionado en el primer <tt>Spinner</tt>
         */
        void onItemSelectedListener(Spinner spinner1, Spinner spinner2);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "HECHO" en <tt>NuevaMedicionFragment</tt>.
         *
         * <p>Se encarga de:
         *
         * 1.- Verificar que el usuario ha introducido toda la información obligatoria (marcada con *).
         * 2.- Enviar la información al Modelo para que se introduzca en la Base de Datos.
         * 3.- Solicitar a la Vista (<tt>NuevaMedicionFragment</tt>) que muestre un mensaje dependiendo
         * de si la inserción en la Base de Datos ha sido exitosa o no.
         * 4.- Si la inserción se realizó con éxito, borrar el <tt>Fragment</tt> actual y mostrar el
         * perfil del paciente actualizado con la nueva medición ya insertada.</p>
         *
         * @param nombre Nombre del paciente
         * @param id ID del paciente
         * @param fecha_hora Fecha y hora de la medición
         * @param joint_side Lado en el que se encuentra la articulación (o columna vertebral)
         * @param joint Articulación sobre la que se ha realizado la medición
         * @param joint_other Articulación (en caso de haber seleccionado la opción "Otra" en el
         *                    <tt>Spinner</tt> joint
         * @param movement Movimiento que ha realizado la articulación
         * @param movement_other Movimiento en caso de haber seleccionado la opción "Otro" en el
         *                    <tt>Spinner</tt> movement
         * @param movement_type Modo en el que se ha realizado el movimiento (con/sin ayuda)
         * @param movement_type_other Modo en caso de haber seleccionado la opción "Otro" en el
         *                            <tt>Spinner</tt> movement_type
         * @param lectura_goniometro Lectura final del goniómetro (angulo de desplazamiento)
         */
        void onDoneClicked(String nombre,
                           int id,
                           String fecha_hora,
                           String joint_side,
                           String joint,
                           String joint_other,
                           String movement,
                           String movement_other,
                           String movement_type,
                           String movement_type_other,
                           String lectura_goniometro);
    }

    /**
     * Interfaz correspondiente al Presentador <tt>NuevoPacientePresenter</tt>
     */
    interface NuevoPacienteFunctions {

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * "GUARDAR" en <tt>NuevoPacienteFragment</tt>.
         * <p>
         * <p>Se encarga de:
         * <p>
         * 1.- Verificar que el usuario ha introducido toda la información obligatoria (marcada con
         * *).
         * 2.- Filtrar la información a un formato que se pueda utilizar para introducirla en la
         * Base de Datos.
         * 3.- Enviar la información, una vez filtrada, al Modelo para que sea introducida en la
         * Base de Datos.
         * 4.- Solicitar a la Vista (<tt>NuevoPacienteFragment</tt>) que muestre un mensaje
         * dependiendo de si la inserción en la Base de Datos ha sido exitosa o no.
         * 5.- Si la inserción se realizó con éxito, borrar el <tt>Fragment</tt> actual y mostrar el
         * listado de pacientes actualizado con el nuevo paciente ya almacenado.</p>
         *
         * @param nombre                  Nombre del paciente (Obligatorio)
         * @param edad                    Edad del paciente (Obligatorio)
         * @param sexo                    Sexo del paciente (Obligatorio)
         * @param id                      ID del paciente (Opcional)
         * @param telefono                Teléfono del paciente (Opcional)
         * @param direccion               Dirección del paciente (Opcional)
         * @param sintomas                Síntomas del paciente (Opcional)
         * @param diagnostico             Diagnóstico del paciente (Obligatorio)
         * @param tratamiento_previo      Tratamiento previo que ha estado siguiendo el paciente
         *                                (Opcional)
         * @param tratamiento_actual      Tratamiento que esta siguiendo actualmente el paciente
         *                                (Opcional)
         * @param comentarios_adicionales Comentarios adicionales que se necesiten introducir
         *                                (Opcional)
         * @param tags                    Etiquetas para recuperar facilmente la información
         *                                del paciente al buscarlo en el listado de pacientes
         *                                (Obligatorio)
         */
        void onSaveClicked(String nombre,
                           String edad,
                           int sexo,
                           String id,
                           String telefono,
                           String direccion,
                           String sintomas,
                           String diagnostico,
                           String tratamiento_previo,
                           String tratamiento_actual,
                           String comentarios_adicionales,
                           String tags);
    }

    /**
     * Interfaz correspondiente al Presentador <tt>PacientesPresenter</tt>
     */
    interface PacientesFunctions {

        /**
         * Método que realiza las operaciones iniciales cuando se crea el
         * <tt>Fragment PacientesFragment</tt>.
         *
         * <p>Se encarga de marcar el cuarto ítem del <tt>NavigationView</tt> (correspondiente a
         * <tt>PacientesFragment</tt>), fijar la orientación de la pantalla en "Portrait" y de
         * utilizar el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>.</p>
         *
         * <p>En este caso, sólo se mostrara el <tt>Toolbar</tt> en el caso de que el usuario no
         * vaya a almacenar una nueva medición en el sistema. Esto es así, para evitar que el
         * usuario vaya a otras partes de la aplicación, a través del mismo, sin haber completado
         * la tarea de almacenar la medición, lo que puede provocar problemas de comportamiento de
         * la aplicación.</p>
         *
         * <p>Así, ademas, se guía al usuario paso a paso, lo que facilita el uso de la aplicación.</p>
         *
         * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
         */
        void setUpInitialSettings(Toolbar toolbar);

        /**
         * Método que se encarga de hacer una configuración inicial del <tt>RecyclerView</tt> que
         * muestra el listado de pacientes almacenados en la Base de Datos.
         *
         * <p>Realiza una operación para mejorar el rendimiento del mismo, así como añadirle el
         * <tt>LayoutManager</tt> y el <tt>RecyclerView.Adapter</tt>.</p>
         *
         * @param recyclerView <tt>RecyclerView</tt> a configurar
         * @param args <tt>Bundle</tt> con los argumentos para crear el <tt>PacienteListAdapter</tt>
         */
        void configRecyclerView(RecyclerView recyclerView, Bundle args);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * para añadir un nuevo paciente en el <tt>PacientesFragment</tt>.
         *
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>NuevoPacienteFragment</tt>.
         */
        void onAñadirPacienteClicked();

        /**
         * Método que se encarga de crear un <tt>ArrayAdapter</tt> para el <tt>Spinner</tt> recibido
         * como parámetro y adjunta un recurso de tipo layout que define cómo se muestra la opción
         * seleccionada en el control del mismo.
         *
         * <p>Ademas, especifica el layout que el <tt>Adapter</tt> usara para mostrar la lista de
         * opciones del <tt>Spinner</tt>.</p>
         *
         * <p>Por último, aplica el adaptador al <tt>Spinner</tt> en cuestión.</p>
         *
         * @param spinner <tt>Spinner</tt> sobre el que se aplica
         */
        void loadSpinner(Spinner spinner);

        /**
         * Método que se encarga de re-asignar el <tt>RecyclerView.Adapter</tt> al
         * <tt>RecyclerView</tt> de <tt>PacientesFragment</tt>, encargado de mostrar el listado de
         * pacientes que coinciden con los criterios de búsqueda (o todos en caso de no especificar
         * ninguno).
         *
         * <p>Primero, obtiene, a partir del Modelo, el listado de pacientes filtrado, según el
         * criterio seleccionado por el usuario en el <tt>Spinner</tt>, y la información introducida
         * por el mismo en el <tt>EditText</tt>.</p>
         *
         * <p>Una vez que se tiene el listado de pacientes ya filtrado, se crea el nuevo
         * <tt>RecyclerView.Adapter</tt> y se asigna al <tt>RecyclerView</tt> recibido como
         * parámetro.</p>
         *
         * @param recyclerView <tt>RecyclerView</tt> sobre el que se aplica este método
         * @param texto Información introducida por el usuario en el cuadro de búsqueda del
         *              <tt>EditText</tt> de <tt>PacientesFragment</tt>
         * @param filtro Filtro seleccionado por el usuario en el <tt>Spinner</tt> de
         *               <tt>PacientesFragment</tt>
         * @param args <tt>Bundle</tt> con los argumentos para crear el <tt>PacienteListAdapter</tt>
         */
        void onTextChanged(RecyclerView recyclerView, String texto, String filtro, Bundle args);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre un elemento
         * del <tt>RecyclerView</tt> que muestra el listado de pacientes en
         * <tt>PacientesFragment</tt>.
         *
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>PerfilPacienteFragment</tt>.
         *
         * @param nombrePaciente Nombre del paciente sobre el que el usuario ha pulsado
         * @param idPaciente ID del paciente sobre el que el usuario ha pulsado
         * @param args <tt>Bundle</tt> con los argumentos para crear el
         *             <tt>PerfilPacienteFragment</tt>
         */
        void onLayoutClicked(String nombrePaciente, int idPaciente, Bundle args);
    }

    /**
     * Interfaz correspondiente al Presentador <tt>PerfilPacientePresenter</tt>
     */
    interface PerfilPacienteFunctions {

        /**
         * Método que se encarga de enviar a la Vista (<tt>PerfilPacienteFragment</tt>) los nombres
         * de las constantes para obtener los parámetros de tipo <tt>Bundle</tt> recibidos en la
         * creación del <tt>Fragment</tt>.
         */
        void onCreate();

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * para añadir una nueva medición en el <tt>PerfilPacienteFragment</tt>.
         * <p>
         * Crea un nuevo <tt>Fragment</tt> de tipo <tt>NuevaMedicionFragment</tt>.
         *
         * @param nombrePaciente    Nombre del paciente al que se le quiere añadir la medición
         * @param idPaciente ID del paciente al que se le quiere añadir la medición
         * @param lecturaGoniometro Lectura del goniómetro obtenida durante la medición
         * @param fechaHora Fecha y hora en la que se realizó la medición
         */
        void onAlmacenarMedicionClicked(String nombrePaciente, int idPaciente,
                                        double lecturaGoniometro, String fechaHora);

        /**
         * Método que se encarga de obtener la información relativa a un paciente a partir del nombre
         * y el id del mismo, mediante una consulta a la Base de Datos.
         *
         * Posteriormente, envía esta información a la Vista (<tt>PerfilPacienteFragment</tt>) para
         * que se la muestre al usuario.
         *
         * @param nombre Nombre del paciente
         * @param id ID del paciente
         */
        void mostrarInformacionPaciente(String nombre, int id);

        /**
         * Método que se encarga de obtener todas las mediciones que se han realizado sobre el
         * paciente en cuestión, introducirlas en objetos de tipo <tt>TableRow</tt> dandoles un
         * formato adecuado, para que se puedan mostrar correctamente en la pantalla del dispositivo,
         * y enviarlas de vuelta a la Vista (<tt>PerfilPacienteFragment</tt>) para que las añada al
         * objeto <tt>TableLayout</tt> que se utiliza para mostrar las mediciones del paciente.
         *
         * @param nombre Nombre del paciente del que recuperar las mediciones
         * @param id ID del paciente del que recuperar las mediciones
         */
        void mostrarMedicionesPaciente(String nombre, int id);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * para borrar el perfil del paciente en el <tt>PerfilPacienteFragment</tt>.
         *
         * <p>Se encarga de crear un dialogo que envía a la Vista para que se lo muestre al usuario,
         * con el fin de confirmar que quiere borrar el paciente junto con todas las mediciones que
         * se le han realizado.</p>
         *
         * <p>Si el usuario confirma que desea eliminar el perfil del paciente, el método se encarga
         * de borrar el <tt>Fragment</tt> actual y mostrar el listado de pacientes actualizado con el
         * paciente ya eliminado.</p>
         *
         * @param nombre Nombre del paciente a eliminar
         * @param id ID del paciente a eliminar
         */
        void onDeletePatientClicked(String nombre, int id);

        /**
         * Método que se encarga de notificar a la Vista (<tt>PerfilPacienteFragment</tt>) si es
         * necesario mostrar el botón para añadir una nueva medición al paciente actual o no.
         */
        void onCreateView();
    }

    /**
     * Interfaz correspondiente al Presentador <tt>UltimasMedicionesPresenter</tt>
     */
    interface UltimasMedicionesFunctions {

        /**
         * Método que realiza las operaciones iniciales cuando se crea el
         * <tt>Fragment UltimasMedicionesFragment</tt>.
         * <p>
         * <p>Se encarga de marcar el tercer ítem del <tt>NavigationView</tt> (correspondiente a
         * <tt>UltimasMedicionesFragment</tt>), fijar la orientación de la pantalla en "Portrait" y
         * de utilizar el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>.</p>
         *
         * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
         */
        void setUpInitialSettings(Toolbar toolbar);

        /**
         * Método que se encarga de hacer una configuración inicial del <tt>RecyclerView</tt> que
         * muestra el listado de las últimas 10 mediciones almacenadas en la Base de Datos.
         *
         * <p>Realiza una operación para mejorar el rendimiento del mismo, así como añadirle el
         * <tt>LayoutManager</tt> y el <tt>RecyclerView.Adapter</tt>.</p>
         *
         * @param recyclerView <tt>RecyclerView</tt> a configurar
         */
        void configRecyclerView(RecyclerView recyclerView);

        /**
         * Evento <tt>ClickListener</tt> que se llama cuando el usuario ha pulsado sobre el botón
         * para borrar la medición en <tt>UltimasMedicionesFragment</tt>.
         *
         * <p>Se encarga de crear un dialogo que envía a la Vista para que se lo muestre al usuario,
         * con el fin de confirmar que quiere borrar la medición seleccionada.</p>
         *
         * <p>Si el usuario confirma que desea eliminar la medición, el método se encarga de mostrar
         * el listado de las últimas 10 mediciones almacenadas, actualizado con la medición ya
         * eliminada.</p>
         *
         * @param nombrePaciente Nombre del paciente al que se le realizó la medición
         * @param fechaHora Fecha y hora en la que se realizó la medición
         */
        void onDeleteMeasurementClicked(String nombrePaciente, String fechaHora);
    }

    /**
     * Interfaz correspondiente al Presentador <tt>SobreAppPresenter</tt>
     */
    interface SobreAppFunctions {
        /**
         * Método que realiza las operaciones iniciales cuando se crea el
         * <tt>Fragment SobreAppFragment</tt>.
         *
         * <p>Se encarga de marcar el quinto ítem del <tt>NavigationView</tt> (correspondiente a
         * <tt>SobreAppFragment</tt>), fijar la orientación de la pantalla en "Portrait",
         * utilizar el <tt>Toolbar</tt> recibido como <tt>SupportActionBar</tt>, y, a partir del icono
         * principal de la aplicación (imagen con formato rectangular), transformarlo en una imagen con
         * un borde circular y enviarselo a la Vista (<tt>SobreAppFragment</tt>) para que se lo asigne
         * al objeto <tt>ImageView</tt> que contiene el icono de la aplicación, mediante el uso de la
         * clase <tt>RoundedBitmapDrawable</tt>.</p>
         *
         * @param toolbar <tt>Toolbar</tt> a emplear como <tt>SupportActionBar</tt>
         */
        void setUpInitialSettings(Toolbar toolbar);
    }
}
