#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Sun Mar  9 19:37:38 2025

@author: pablo
"""

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
from sklearn.preprocessing import MinMaxScaler
from tensorflow.keras.models import Model
from tensorflow.keras.layers import LSTM, Dense, RepeatVector, TimeDistributed, Input
from tensorflow.keras.optimizers import Adam
from sklearn.metrics import mean_squared_error


def train_lstm_autoencoder6(file_path, time_steps=10, latent_dim=32, lstm_units=[128, 64, 32], epochs=10, batch_size=32):
    # Cargar datos
    df = pd.read_csv(file_path)
    
    # Normalización de los datos
    scaler = MinMaxScaler()
    data_scaled = scaler.fit_transform(df)
    
    # Crear secuencias de datos
    X = []
    for i in range(len(data_scaled) - time_steps):
        X.append(data_scaled[i:i+time_steps])
    X = np.array(X)
    Xmin=X.min(axis=0)
    Xmax=X.max(axis=0)
    
    # Definición del Autoencoder LSTM
    input_dim = X.shape[2]
    input_layer = Input(shape=(time_steps, input_dim))
    
    # Autoencoder LSTM Mejorado con más capas y unidades
    encoded = LSTM(lstm_units[0], activation='selu', return_sequences=True)(input_layer)
    encoded = LSTM(lstm_units[1], activation='selu', return_sequences=True)(encoded)
    encoded = LSTM(lstm_units[2], activation='selu', return_sequences=False)(encoded)

    decoded = RepeatVector(time_steps)(encoded)
    decoded = LSTM(lstm_units[2], activation='selu', return_sequences=True)(decoded)
    decoded = LSTM(lstm_units[1], activation='selu', return_sequences=True)(decoded)
    decoded = LSTM(lstm_units[0], activation='selu', return_sequences=True)(decoded)
    decoded = TimeDistributed(Dense(input_dim))(decoded)
    
    
    # Modelo Autoencoder
    autoencoder = Model(input_layer, decoded)
    autoencoder.compile(optimizer=Adam(learning_rate=0.001), loss='mse')
    
    # Entrenar el modelo
    history = autoencoder.fit(X, X, epochs=epochs, batch_size=batch_size, validation_split=0.2, verbose=1)
    
    # Definir modelo encoder
    encoder = Model(input_layer, encoded)
    
    # Visualizar pérdidas del entrenamiento
    plt.figure()
    plt.plot(history.history['loss'], label='Train Loss')
    plt.plot(history.history['val_loss'], label='Validation Loss')
    plt.xlabel('Epochs')
    plt.ylabel('Loss')
    plt.title('Loss del modelo')
    plt.legend()
    plt.grid()
    plt.show()
    
    # Predicción y cálculo del error
    X_pred = autoencoder.predict(X)
    mse = np.mean(np.mean((X_pred - X) ** 2, axis=1), axis=1)
    
    # Umbral de fallo (percentil 95)
    thresh = np.percentile(mse, 95)
    anomalies = mse > thresh
    
    # Representación de los valores T² y Q
    h = encoder.predict(X)
    hm = h.mean(axis=0)
    hdesv = np.cov(h.T)
    
    # Para evitar problemas de singularidad
    hdesv += np.eye(hdesv.shape[0]) * 1e-5
    covin = np.linalg.inv(hdesv)
    
    T2 = np.array([np.dot(np.dot((h[i] - hm), covin), (h[i] - hm).T) for i in range(h.shape[0])])
    UmbralT2 = np.percentile(T2, 99)
    
    plt.figure(figsize=(10, 6))
    plt.plot(range(1, len(T2) + 1), T2, label='T^2', color='blue')
    plt.axhline(y=UmbralT2, color='red', linestyle='--', label='Umbral')
    plt.xlabel('Observaciones')
    plt.ylabel('Valor de T^2')
    plt.title('Vector T^2 y Umbral')
    plt.legend()
    plt.grid()
    plt.show()
    
    res = X - X_pred
    residuo = res.reshape(res.shape[0], -1)
    rmed = residuo.mean(axis=0)
    rcov = np.cov(residuo.T)
    
    # Para evitar problemas de singularidad
    rcov += np.eye(rcov.shape[0]) * 1e-5
    rcovin = np.linalg.inv(rcov)
    
    Q = np.array([np.dot(np.dot((residuo[i] - rmed), rcovin), (residuo[i] - rmed).T) for i in range(h.shape[0])])
    UmbralQ = np.percentile(Q, 98)
    
    plt.figure(figsize=(10, 6))
    plt.plot(range(1, len(Q) + 1), Q, label='Q', color='blue')
    plt.axhline(y=UmbralQ, color='red', linestyle='--', label='Umbral')
    plt.xlabel('Observaciones')
    plt.ylabel('Valor de Q')
    plt.title('Vector Q y Umbral')
    plt.legend()
    plt.grid()
    plt.show()
    
    return autoencoder, encoder, history, Xmin, Xmax, UmbralT2, UmbralQ, hm, hdesv, rcov, rmed

# Uso:
# autoencoder, encoder, history = train_lstm_autoencoder('datos_planta.csv')

autoencoder, encoder, history, Xmin, Xmax, UmbralT2, UmbralQ, hm, hdesv, rcov, rmed = train_lstm_autoencoder6('/Users/pablo/Documents/fault_free_testing.csv')

#/Users/pablo/Documents/IEIA/TFG/DOCUMENTACION/datos_csv/d00_te.csv
np.savez('/Users/pablo/Documents/IEIA/TFG/Autoencoders/AutoencoderLSTM6_Data.npz', 
        Xmin=Xmin, Xmax=Xmax, UmbralT2=UmbralT2, UmbralQ=UmbralQ, 
        hm=hm,hdesv=hdesv,rcov=rcov,rmed=rmed)

# Guardar los modelos
autoencoder.save('/Users/pablo/Documents/IEIA/TFG/MODELOS/autoencoder_LSTM6.keras')
encoder.save('/Users/pablo/Documents/IEIA/TFG/MODELOS/encoder_LSTM6.keras')


print("Modelos guardados.")


