#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Nov 20 19:18:58 2025

@author: pablo
"""

import os
from keras.models import load_model
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt

# -----------------------------
# Cargar modelos y datos NPZ
# -----------------------------
archivo_npz = np.load('/Users/pablo/Documents/IEIA/TFG/Autoencoders/AutoencoderData.npz')

Xmin = archivo_npz['Xmin']
Xmax = archivo_npz['Xmax']
UmbralT2 = archivo_npz['UmbralT2']
UmbralQ = archivo_npz['UmbralQ']
hm = archivo_npz['hm']
rmed = archivo_npz['rmed']
rcov = archivo_npz['rcov']
hdesv = archivo_npz['hdesv']

rcovin = np.linalg.inv(rcov)
covin = np.linalg.inv(hdesv)

# Cargar modelos
directorio_modelos = '/Users/pablo/Documents/IEIA/TFG/MODELOS'
autoencoder = load_model(os.path.join(directorio_modelos, 'autoencoder_model.keras'))
encoder = load_model(os.path.join(directorio_modelos, 'encoder_model.keras'))


# ---------------------------------------------------------------
# FUNCIÓN PRINCIPAL PARA PROCESAR CADA ESCENARIO
# ---------------------------------------------------------------
def procesar_escenario(num_escenario):
    esc = str(num_escenario).zfill(2)
    ruta = f'/Users/pablo/Documents/IEIA/TFG/datos_csv/d{esc}_te.csv'
    X = pd.read_csv(ruta)
    Xn = (X - Xmin) / (Xmax - Xmin)

    # Inferencia
    h = encoder.predict(Xn)
    salida = autoencoder.predict(Xn)

    residuo = (salida - Xn).values  # shape: (n_obs, n_vars)

    # ---------------------- T² ----------------------
    T2 = np.array([ (h[i] - hm) @ covin @ (h[i] - hm).T for i in range(len(h)) ])

    # ---------------------- Q -----------------------
    Q = np.array([ (residuo[i] - rmed) @ rcovin @ (residuo[i] - rmed).T for i in range(len(residuo)) ])

    # ---------------------- Detectar fallo (10 consecutivos) ------------------
    def detectar_fallo(vector, umbral, consecutivos=10):
        contador = 0
        for i, v in enumerate(vector):
            if v > umbral:
                contador += 1
            else:
                contador = 0
            if contador == consecutivos:
                return i - consecutivos + 1
        return None

    t_fallo_T = detectar_fallo(T2, UmbralT2, consecutivos=10)
    t_fallo_Q = detectar_fallo(Q, UmbralQ, consecutivos=10)

    # ---------------------- Métricas ----------------
    FA_T = np.sum(T2[:160] > UmbralT2) / 160 * 100
    AL_T = np.sum(T2[160:] > UmbralT2) / (len(T2) - 160) * 100 if len(T2) > 160 else 0.0

    FA_Q = np.sum(Q[:160] > UmbralQ) / 160 * 100
    AL_Q = np.sum(Q[160:] > UmbralQ) / (len(Q) - 160) * 100 if len(Q) > 160 else 0.0

    # ---------------------- Contribuciones (residuo^2 en t_fallo_Q) ----------------
    if t_fallo_Q is not None:
        contrib = np.square(residuo[t_fallo_Q, :])  # array length = n_vars
    else:
        contrib = None  # lo manejaremos al crear el DataFrame

    return {
        "escenario": num_escenario,
        "FA_T": FA_T,
        "AL_T": AL_T,
        "t_F_T": t_fallo_T,
        "FA_Q": FA_Q,
        "AL_Q": AL_Q,
        "t_F_Q": t_fallo_Q,
        "contribuciones": contrib,
        "n_vars": X.shape[1]
    }


# ---------------------------------------------------------------
# PROCESAR TODOS LOS ESCENARIOS
# ---------------------------------------------------------------
resultados = []
contribuciones_por_escenario = {}
n_vars_esperadas = None

for esc in range(1, 22):
    print(f"Procesando escenario {esc}...")
    r = procesar_escenario(esc)
    resultados.append(r)
    contribuciones_por_escenario[esc] = r["contribuciones"]
    if n_vars_esperadas is None:
        n_vars_esperadas = r["n_vars"]

# ---------------------------------------------------------------
# CONSTRUIR DataFrame DE CONTRIBUCIONES: filas=variables (1..n_vars), columnas=Escenario_1..Escenario_21
# ---------------------------------------------------------------
# Inicializamos con NaN
cols = [f'Escenario_{i}' for i in range(1, 22)]
df_contrib = pd.DataFrame(index=range(1, n_vars_esperadas+1), columns=cols, dtype=float)

for esc in range(1, 22):
    contrib = contribuciones_por_escenario[esc]
    col = f'Escenario_{esc}'
    if contrib is None:
        # dejamos NaN para todas las variables de ese escenario
        df_contrib[col] = np.nan
    else:
        # asegúrate de que la longitud coincide con n_vars_esperadas
        if len(contrib) != n_vars_esperadas:
            # si hay discrepancia, recortamos o rellenamos con NaN
            if len(contrib) > n_vars_esperadas:
                df_contrib[col] = contrib[:n_vars_esperadas]
            else:
                tmp = np.full(n_vars_esperadas, np.nan, dtype=float)
                tmp[:len(contrib)] = contrib
                df_contrib[col] = tmp
        else:
            df_contrib[col] = contrib

# Insertar columna 'Variable' con índices 1..n_vars
df_contrib.insert(0, 'Variable', range(1, n_vars_esperadas+1))

# ---------------------------------------------------------------
# CREAR DataFrame RESUMEN
# ---------------------------------------------------------------
df_resumen = pd.DataFrame([
    {
        "Escenario": r["escenario"],
        "Falsas Alarmas T2 (%)": r["FA_T"],
        "Alarmas Detectadas T2 (%)": r["AL_T"],
        "t_fallo_T2": r["t_F_T"],
        "Falsas Alarmas Q (%)": r["FA_Q"],
        "Alarmas Detectadas Q (%)": r["AL_Q"],
        "t_fallo_Q": r["t_F_Q"],
    }
    for r in resultados
])

# ---------------------------------------------------------------
# EXPORTAR A EXCEL: Hoja 'Resumen' y hoja 'Contribuciones'
# ---------------------------------------------------------------
nombre_archivo = 'Resultados_AE_Contribuciones_Columnas.xlsx'
directorio_salida = '/Users/pablo/Documents/IEIA/TFG/Autoencoders'
ruta_salida = os.path.join(directorio_salida, nombre_archivo)

with pd.ExcelWriter(ruta_salida, engine='openpyxl') as writer:
    df_resumen.to_excel(writer, sheet_name='Resumen', index=False)
    df_contrib.to_excel(writer, sheet_name='Contribuciones', index=False)

print("Excel generado en:", ruta_salida)
