import numpy as np

def calcula_BICs(t2, t2a, q, qa, alpha_BIC=0.5):
    '''
    Calcula los coeficientes de inferencia bayesiana para la implementación distribuida en bloques.
    Entradas: 
    - t2: array [n_observaciones x n_bloques] (o su transpuesta; se normaliza internamente)
    - t2a: UCL de T². Vector de n_bloques
    - q: array [n_observaciones x n_bloques]
    - qa: UCL de Q. Vector de n_bloques
    - alpha_BIC: prior P_N (por defecto 0.5)
    Salidas:
    - bic_t2: vector [n_observaciones] combinando la información T2 de todos los bloques
    - bic_q: vector [n_observaciones] combinando la información Q de todos los bloques
    - umbral_bic_t2: umbral (percentil 99) para bic_t2
    - umbral_bic_q: umbral (percentil 99) para bic_q
    '''
    # seguridad y conversión de tipos
    t2 = np.array(t2, dtype=float)
    q = np.array(q, dtype=float)
    t2a = np.array(t2a, dtype=float)
    qa = np.array(qa, dtype=float)

    # Aceptar entradas en forma (n_obs, n_bloques) o (n_bloques, n_obs)
    if t2.shape[0] != t2a.shape[0] and t2.shape[1] == t2a.shape[0]:
        # t2 viene en forma (n_obs, n_bloques) -> transpongo a (n_bloques, n_obs)
        t2 = t2.T
    if q.shape[0] != qa.shape[0] and q.shape[1] == qa.shape[0]:
        q = q.T

    n_bloque, n = t2.shape  # ahora t2 es (n_bloque, n_obs)

    P_N = float(alpha_BIC)
    P_F = 1.0 - P_N

    # protecciones numéricas
    eps = 1e-12
    t2_safe = np.maximum(t2, eps)
    q_safe = np.maximum(q, eps)
    t2a_safe = np.maximum(t2a, eps)
    qa_safe = np.maximum(qa, eps)

    # inicialización
    P_xiN_T2 = np.zeros((n_bloque, n), dtype=float)
    P_xiF_T2 = np.zeros((n_bloque, n), dtype=float)
    P_xi_T2 = np.zeros((n_bloque, n), dtype=float)
    P_Fxi_T2 = np.zeros((n_bloque, n), dtype=float)

    P_xiN_Q = np.zeros((n_bloque, n), dtype=float)
    P_xiF_Q = np.zeros((n_bloque, n), dtype=float)
    P_xi_Q = np.zeros((n_bloque, n), dtype=float)
    P_Fxi_Q = np.zeros((n_bloque, n), dtype=float)

    # cálculo probabilidades bloque a bloque (vectorizado por columnas en bucle)
    for jj in range(n_bloque):
        # Para T2:
        # P(x|N) ≈ exp(-t2 / t2a)
        P_xiN_T2[jj, :] = np.exp(- t2[jj, :] / (t2a_safe[jj] + eps))
        # P(x|F) ≈ exp(-t2a / t2)
        P_xiF_T2[jj, :] = np.exp(- t2a_safe[jj] / (t2_safe[jj, :] + eps))

        # mezcla y prob posterior P(F|x)
        P_xi_T2[jj, :] = P_N * P_xiN_T2[jj, :] + P_F * P_xiF_T2[jj, :]
        # evitar divisiones por 0
        P_xi_T2[jj, :] = np.clip(P_xi_T2[jj, :], eps, np.inf)
        P_Fxi_T2[jj, :] = (P_F * P_xiF_T2[jj, :]) / P_xi_T2[jj, :]

        # Para Q (misma lógica)
        P_xiN_Q[jj, :] = np.exp(- q[jj, :] / (qa_safe[jj] + eps))
        P_xiF_Q[jj, :] = np.exp(- qa_safe[jj] / (q_safe[jj, :] + eps))
        P_xi_Q[jj, :] = P_N * P_xiN_Q[jj, :] + P_F * P_xiF_Q[jj, :]
        P_xi_Q[jj, :] = np.clip(P_xi_Q[jj, :], eps, np.inf)
        P_Fxi_Q[jj, :] = (P_F * P_xiF_Q[jj, :]) / P_xi_Q[jj, :]

    # suma de P_xiF sobre bloques (por muestra), shape -> (n,)
    sum_P_xiF_T2 = np.sum(P_xiF_T2, axis=0)
    sum_P_xiF_Q = np.sum(P_xiF_Q, axis=0)

    # variables intermedias para construir BIC
    aa = P_xiF_T2 * P_Fxi_T2     # (n_bloque, n)
    ab = P_xiF_Q * P_Fxi_Q       # (n_bloque, n)

    # normalizar por la suma global por muestra (evitar dividir por 0)
    sum_P_xiF_T2_safe = np.clip(sum_P_xiF_T2, eps, np.inf)
    sum_P_xiF_Q_safe = np.clip(sum_P_xiF_Q, eps, np.inf)

    aaa = aa / sum_P_xiF_T2_safe   # (n_bloque, n)
    aab = ab / sum_P_xiF_Q_safe   # (n_bloque, n)

    # BIC agregados por muestra
    bic_t2 = np.sum(aaa, axis=0)  # (n,)
    bic_q = np.sum(aab, axis=0)   # (n,)

    # umbrales estadísticos (percentil 99)
    umbral_bic_t2 = np.percentile(bic_t2, 99)
    umbral_bic_q = np.percentile(bic_q, 99)

    return bic_t2, bic_q, umbral_bic_t2, umbral_bic_q
