#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Mar  7 20:25:41 2025

@author: pablo
"""

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
from sklearn.preprocessing import MinMaxScaler
from tensorflow.keras.models import Model
from tensorflow.keras.layers import LSTM, Dense, RepeatVector, TimeDistributed, Input
from tensorflow.keras.optimizers import Adam

# Carga y pretratamiento de datos:
df = pd.read_csv('/Users/pablo/Documents/fault_free_testing.csv')

#Usamos 240k valores y quitamos las 3 primeras columnas que no aportan informacion
X=df.iloc[:240000, 3:]

# Normalización de los datos
# scaler = MinMaxScaler(X)
# Xn = scaler.fit_transform()  # Conversion a ndarray de numpy
Xmin = X.min(axis=0).values  # Mínimos de cada variable
Xmax = X.max(axis=0).values  # Máximos de cada variable
Xn=(X-Xmin)/(Xmax-Xmin)


def train_lstm_autoencoder(Xn, time_steps=5, lstm_units=[48, 24], epochs=4, batch_size=32):
    
    """
    Entrena un autoencoder basado en LSTM con los datos proporcionados.
    
    Parámetros:
    -----------
    Xn : numpy.ndarray
        Matriz de entrada con la secuencia de datos normalizados para entrenar el modelo LSTM.
    time_steps : int, opcional
        Numero de datos a usar como ventana deslizante para capturar dependencias a largo plazo(por defecto 5)
    lstm_units : list, optional
        Numero de neuronas en cada una de las dos capas (por defecto 48 y 24)
    epochs : int, opcional
        Número de épocas para entrenar el modelo (por defecto 4).
    batch_size : int, opcional
        Tamaño del batch en cada iteración del entrenamiento (por defecto 32).
    
    Retorna:
    --------
    history : keras.callbacks.History
        Objeto que contiene la historia del entrenamiento, incluyendo la pérdida (`loss`) y la pérdida de validación (`val_loss`).
    autoencoder : keras.
        Modelo del autoencoder (decoder)
    encoder: keras.
        Modelo del encoder
    T2, uT2, Q, uQ, hm, hdesv
        variables estadisticas para deteccion de fallos
    time_steps: 
        numero de pasos temporales de la ventana deslizante que emplea la LSTM
    
    
    Ejemplo de uso:
    ---------------
    >>> autoencoder, encoder, history, T2, uT2, Q, uQ, hm, hdesv, time_steps = train_lstm_autoencoder(Xn)
    """
   
    # Crear secuencias de datos adaptadas a la red LSTM a partir de la matriz de datos normalizada
    Xn_lstm = []
    for i in range(len(Xn) - time_steps):
        Xn_lstm.append(Xn[i:i+time_steps])
    Xn_lstm = np.array(Xn_lstm)
    
    # Definición del Autoencoder LSTM
    input_dim = Xn_lstm.shape[2]
    input_layer = Input(shape=(time_steps, input_dim))

    # Codificador
    encoded = LSTM(lstm_units[0], activation='selu', return_sequences=True)(input_layer)
    encoded = LSTM(lstm_units[1], activation='selu', return_sequences=False)(encoded)
    
    # Decodificador
    decoded = RepeatVector(time_steps)(encoded)
    decoded = LSTM(lstm_units[1], activation='selu', return_sequences=True)(decoded)
    decoded = LSTM(lstm_units[0], activation='selu', return_sequences=True)(decoded)
    decoded = TimeDistributed(Dense(input_dim))(decoded)
    
    # Modelo Autoencoder
    autoencoder = Model(input_layer, decoded)
    autoencoder.compile(optimizer=Adam(learning_rate=0.001), loss='mse')
    
    # Entrenamiento del modelo
    history = autoencoder.fit(Xn_lstm, Xn_lstm, epochs=epochs, batch_size=batch_size, validation_split=0.2, verbose=1)
    
    # Definir modelo encoder
    encoder = Model(input_layer, encoded)
    
    # Visualizar pérdidas del entrenamiento
    plt.figure()
    plt.plot(history.history['loss'], label='Train Loss')
    plt.plot(history.history['val_loss'], label='Validation Loss')
    plt.xlabel('Epochs')
    plt.ylabel('Loss')
    plt.title('Loss del modelo')
    plt.legend()
    plt.grid()
    plt.show()
    
    # Predicción y cálculo del error
    X_pred = autoencoder.predict(Xn_lstm)
    
    # #(no usado) Cálculo del error cuadrático medio 
    # mse = np.mean(np.mean((X_pred - Xn_lstm) ** 2, axis=1), axis=1)
    
    # Representación de los valores T² y Q
    h = encoder.predict(Xn_lstm)
    hm = h.mean(axis=0)
    hdesv = np.cov(h.T)
    
    # Para evitar problemas de singularidad
    hdesv += np.eye(hdesv.shape[0]) * 1e-5
    covin = np.linalg.inv(hdesv)
    
    T2 = np.array([np.dot(np.dot((h[i] - hm), covin), (h[i] - hm).T) for i in range(h.shape[0])])
    uT2 = np.percentile(T2, 99)
    
    plt.figure(figsize=(10, 6))
    plt.plot(range(1, len(T2) + 1), T2, label='T^2', color='blue')
    plt.axhline(y=uT2, color='red', linestyle='--', label='Umbral')
    plt.xlabel('Observaciones')
    plt.ylabel('Valor de T^2')
    plt.title('Vector T^2 y Umbral')
    plt.legend()
    plt.grid()
    plt.show()
    
    
    # Calculamos los residuos para obtener Q
    res = Xn_lstm - X_pred
    residuo = res.reshape(res.shape[0], -1)
    
    # # En este caso no los vamos a usar
    # rmed = residuo.mean(axis=0)
    # rcov = np.cov(residuo.T)
    # # Para evitar problemas de singularidad
    # rcov += np.eye(rcov.shape[0]) * 1e-5
    # rcovin = np.linalg.inv(rcov)
    
    # Emplearemos Q=r*r^t para evitar valoles tan grandes de Q al no dividir por un numero cercano a 0
    # Q = np.array([np.dot(np.dot((residuo[i] - rmed), rcovin), (residuo[i] - rmed).T) for i in range(h.shape[0])])
   
    Q = np.array([np.dot((residuo[i]), (residuo[i]).T) for i in range(h.shape[0])])
    uQ = np.percentile(Q, 99)
    
    plt.figure(figsize=(10, 6))
    plt.plot(range(1, len(Q) + 1), Q, label='Q', color='blue')
    plt.axhline(y=uQ, color='red', linestyle='--', label='Umbral')
    plt.xlabel('Observaciones')
    plt.ylabel('Valor de Q')
    plt.title('Vector Q y Umbral')
    plt.legend()
    plt.grid()
    plt.show()
    
    return autoencoder, encoder, history, T2, uT2, Q, uQ, hm, hdesv, time_steps


# llamamos a la funcion y le pasamos la matriz de datos normalizada
autoencoder, encoder, history, T2, uT2, Q, uQ, hm, hdesv, time_steps = train_lstm_autoencoder(Xn)

plt.plot(history.history['loss'])
plt.plot(history.history['val_loss'])
plt.title('Loss del modelo')
plt.ylabel('Loss')
plt.xlabel('Epoch')
plt.legend(['Train', 'Validation'], loc='upper right')
plt.show()

# # Salvaguarda de ldatos

# np.savez('/Users/pablo/Documents/IEIA/TFG/Autoencoders/AutoencoderLSTM_Data.npz', 
#         Xmin=Xmin, Xmax=Xmax, scaler=scaler, T2=T2, UmbralT2=uT2, Q=Q, UmbralQ=uQ, hm=hm, hdesv=hdesv, time_steps=time_steps)

# autoencoder.save('/Users/pablo/Documents/IEIA/TFG/MODELOS/autoencoder_LSTM.keras')
# encoder.save('/Users/pablo/Documents/IEIA/TFG/MODELOS/encoder_LSTM.keras')


# print("Modelos guardados.")

# columnas_usadas = list (X.colums)
