#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Jun  4 20:44:28 2025

@author: pablo
"""

import os
from keras.models import load_model
import pandas as pd
import numpy as np

# Rutas base
archivo_npz = np.load('/Users/pablo/Documents/IEIA/TFG/Autoencoders/AutoencoderLSTM_Data.npz')
directorio_modelos = '/Users/pablo/Documents/IEIA/TFG/MODELOS'
autoencoder = load_model(os.path.join(directorio_modelos, 'autoencoder_LSTM.keras'))
encoder = load_model(os.path.join(directorio_modelos, 'encoder_LSTM.keras'))

# Cálculo para un escenario
def calcular_valores(escenario):
    
    """
    Calcula las estadísticas T² y Q para un escenario dado de detección de fallos 
    usando un autoencoder LSTM.

    El análisis incluye:
        - Porcentaje de falsas alarmas y alarmas detectadas para T² y Q
        - Momento de detección del fallo (t_fallo) si se detectan 10 observaciones consecutivas fuera de umbral

    Parámetros
    ----------
    escenario : str
        Número de escenario a analizar (por ejemplo, "01", "12", etc.)

    Retorna
    -------
    tuple
        (falsas_alarmas_T2, alarmas_detectadas_T2, t_fallo_T2,
         falsas_alarmas_Q, alarmas_detectadas_Q, t_fallo_Q)
    """
    
    if int(escenario) < 10:
        escenario = escenario.zfill(2)

    ruta = f'/Users/pablo/Documents/IEIA/TFG/datos_csv/d{escenario}_te.csv'
    if not os.path.exists(ruta):
        print(f"❌ Archivo no encontrado: {ruta}")
        return [np.nan] * 6

    X = pd.read_csv(ruta)

    # Extraer datos del .npz
    Xmin = archivo_npz['Xmin']
    Xmax = archivo_npz['Xmax']
    # scaler =  archivo_npz['scaler']
    UmbralT2 = archivo_npz['UmbralT2']
    UmbralQ = archivo_npz['UmbralQ']
    hm = archivo_npz['hm']
    hdesv = archivo_npz['hdesv']
    time_steps = archivo_npz['time_steps']

    # Normalizar
    Xn=(X-Xmin)/(Xmax-Xmin)
    Xn=np.array(Xn)
    
    # Creamos las secuencias de datos adaptadas a la red LSTM a partir de la matriz de datos normalizada
    Xn_lstm = []
    for i in range(len(Xn) - time_steps):
        Xn_lstm.append(Xn[i:i+time_steps])
    Xn_lstm = np.array(Xn_lstm)
    

    # Predicción y cálculo del error (autoencoder)
    X_pred = autoencoder.predict(Xn_lstm)
    # Representación de los valores T² y Q (codificador)
    h = encoder.predict(Xn_lstm)
    # Cálculo de las estadícas para el analisis de componentes principales
    covin = np.linalg.inv(hdesv)

    # T²
    T2 = np.array([np.dot(np.dot(h[i] - hm, covin), (h[i] - hm).T) for i in range(h.shape[0])])

    # Q
    residuo = Xn_lstm - X_pred
    # Calculamos los residuos para obtener Q
    res = Xn_lstm - X_pred
    residuo = res.reshape(res.shape[0], -1)

    #----- Q -----#


    Q = np.array([np.dot((residuo[i]), (residuo[i]).T) for i in range(h.shape[0])])
    print(f"\n📂 Escenario {escenario}")

    # === T2 ===
    Falsas_alarmas_T = np.mean(T2[:160] > UmbralT2) * 100
    Alarmas_detectadas_T = np.mean(T2[160:len(T2)] > UmbralT2) * 100
    t_fallo_T = None
    fallo = 0
    for i in range(len(T2)):
        if T2[i] > UmbralT2:
            fallo += 1
            if fallo == 10:
                t_fallo_T = i - 9
                break
        else:
            fallo = 0
    print("Falsas alarmas (T2):", Falsas_alarmas_T)
    print("Alarmas detectadas (T2):", Alarmas_detectadas_T)
    print("Primera detección fallo T2 en:", t_fallo_T)



    # === Q ===
    Falsas_alarmas_Q = np.mean(Q[:160] > UmbralQ) * 100
    Alarmas_detectadas_Q = np.mean(Q[160:len(Q)] > UmbralQ) * 100
    t_fallo_Q = None
    fallo = 0
    for i in range(len(Q)):
        if Q[i] > UmbralQ:
            fallo += 1
            if fallo == 10:
                t_fallo_Q = i - 9
                break
        else:
            fallo = 0
    print("Falsas alarmas (Q):", Falsas_alarmas_Q)
    print("Alarmas detectadas (Q):", Alarmas_detectadas_Q)
    print("Primera detección fallo Q en:", t_fallo_Q)

    return Falsas_alarmas_T, Alarmas_detectadas_T, t_fallo_T, Falsas_alarmas_Q, Alarmas_detectadas_Q, t_fallo_Q


# Ejecutar para todos los escenarios
valores_por_escenario = []
for escenario in range(1, 22):
    valores = calcular_valores(str(escenario))
    valores_por_escenario.append(valores)

# Crear DataFrame
df_resultados = pd.DataFrame(valores_por_escenario, columns=[
    'FalsasA(T2)', 'Alarmas(T2)', 't_Fallo(T2)',
    'FalsasA(Q)', 'Alarmas(Q)', 't_Fallo(Q)'
])
df_resultados.insert(0, 'Fallo', range(1, 22))

# Guardar a Excel
ruta_salida = '/Users/pablo/Documents/IEIA/TFG/Autoencoders/ResultadosLSTM.xlsx'
df_resultados.to_excel(ruta_salida, index=False)
print(f"\n✅ Resultados guardados en: {ruta_salida}")


